#include "console.h"

#include <util/string/escape.h>
#include <util/string/vector.h>
#include <util/string/join.h>
#include <util/generic/hash.h>

namespace NSecretSearch {
    namespace NOutput {
        namespace {
            constexpr char kColorDefault[] = " \033[0m ";
            constexpr char kColorTitle[] = " \033[95m ";
            constexpr char kColorSecret[] = " \033[91m ";
            constexpr char kColorHighlight[] = " \033[93m ";

            void writeAdditionalInfo(TSecretAdditional additionalInfo, IOutputStream& out) {
                TVector<TStringBuf> keys;
                Transform(additionalInfo.begin(), additionalInfo.end(), std::back_inserter(keys),
                          [](auto const& pair) {
                              return pair.first;
                          });
                Sort(keys);

                for (const auto& key : keys) {
                    if (additionalInfo[key].empty()) {
                        continue;
                    }

                    out << "\n\t\t" << key << ": " << additionalInfo[key];
                }
            }

        }

        void TConsole::Write(const TPathResult& result) {
            out << kColorTitle << result.Path << kColorDefault << Endl;
            ;

            for (const auto& secret : result.SearcherResults) {
                if (!secret.Secret.empty()) {
                    out << "\tSecret: "
                        << kColorSecret
                        << secret.Secret
                        << kColorDefault
                        << Endl;
                }

                if (secret.Type) {
                    out << "\tType: " << secret.Type << Endl;
                }

                if (secret.Validated) {
                    out << "\tValidated:" << kColorSecret << "yes" << kColorDefault << Endl;
                } else {
                    out << "\tValidated: no" << Endl;
                }

                if (!secret.Owners.empty()) {
                    out << "\tOwners:" << kColorHighlight;
                    out << JoinSeq(", ", secret.Owners);
                    out << kColorDefault << Endl;
                }

                if (!secret.Additional.empty()) {
                    out << "\tAdditional:" << kColorHighlight;
                    writeAdditionalInfo(secret.Additional, out);
                    out << kColorDefault << Endl;
                }

                if (secret.SourceLine.empty()) {
                    out << "\tLineNo: " << secret.LineNo << Endl;
                } else {
                    out << "\tLine [" << secret.LineNo << "]: " << secret.SourceLine << Endl;
                }

                out << Endl;
            }
        }

    }
}
