#include "searcher.h"

namespace NSecretSearch {
    namespace {
        constexpr size_t kQueueSize = 4096;

    }

    TSearcher::TSearcher(const TSearchOptions& opts)
        : opts(opts)
        , ctx(opts.InternalCtx())
        , worker(new NSSInternal::TSearchWorker(ctx))
    {
    }

    TMaybe<TSourceResult> TSearcher::CheckSource(ISource& source) {
        return worker->CheckSource(source);
    }

    TVector<TStructResult> TSearcher::CheckStruct(NJson::TJsonValue& root) {
        return worker->CheckStruct(root);
    }

    TVector<TPathResult> TSearcher::CheckPath(const TFsPath& path) {
        NSSInternal::NFileWalker::TWalkerOptions walkerOpts{
            .Excludes = opts.Excludes};

        NSSInternal::NFileWalker::TWalker walker(walkerOpts);
        TVector<TPathResult> results;
        auto walkFn = [this, &results](const TString& path) -> bool {
            auto result = worker->CheckPath(path);
            if (result.Defined()) {
                results.push_back(result.GetRef());
            }
            return true;
        };

        walker.Walk(path, walkFn);

        return results;
    }

    TSearcherThreaded::TSearcherThreaded(const TSearchOptions& opts, size_t numThreads)
        : opts(opts)
        , ctx(opts.InternalCtx())
        , numThreads(numThreads)
    {
        Y_ENSURE(numThreads > 0, "To use not threaded searcher use simple TSearcher instead");
        for (size_t i = 0; i < numThreads; ++i) {
            workers.emplace_front(ctx);
        }
    }

    TMaybe<TSourceResult> TSearcherThreaded::CheckSource(ISource& source) {
        return workers[0].CheckSource(source);
    }

    TVector<TStructResult> TSearcherThreaded::CheckStruct(NJson::TJsonValue& root) {
        return workers[0].CheckStruct(root);
    }

    TVector<TPathResult> TSearcherThreaded::CheckPath(const TFsPath& path) {
        NSSInternal::TJobsQueue jobsQueue(kQueueSize);
        NSSInternal::TPathResultQueue resultsQueue(0);

        auto* threadPool = SystemThreadFactory();
        TDeque<TAutoPtr<IThreadFactory::IThread>> threads;
        for (auto&& worker : workers) {
            worker.ResetQueue(&jobsQueue, &resultsQueue);
            threads.emplace_front(threadPool->Run(&worker));
        }

        // Start search
        NSSInternal::NFileWalker::TWalkerOptions walkerOpts{
            .Excludes = opts.Excludes};

        auto walkFn = [&jobsQueue](const TString& path) -> bool {
            jobsQueue.Push(path);
            return true;
        };

        NSSInternal::NFileWalker::TWalker walker(walkerOpts);
        walker.Walk(path, walkFn);

        // Wait workers
        jobsQueue.Stop();
        for (auto& thread : threads) {
            thread->Join();
        }

        // Parse results
        resultsQueue.Stop();
        TVector<TPathResult> results;
        results.reserve(resultsQueue.Size());
        while (true) {
            auto maybeResult = resultsQueue.Pop();
            if (maybeResult.Empty())
                break;

            results.push_back(maybeResult.GetRef());
        }
        return results;
    }

}
