#include "searcher_stream.h"

namespace NSecretSearch {
    namespace {
        constexpr size_t kQueueSize = 4096;

    }

    TSearcherStream::TSearcherStream(const TSearchOptions& opts, NOutput::IWriter* writer)
        : opts(opts)
        , ctx(opts.InternalCtx())
        , output(writer)
        , worker(new NSSInternal::TSearchWorker(ctx))
    {
    }

    bool TSearcherStream::CheckSource(ISource& source) {
        bool founded = false;
        output->Start();
        auto result = worker->CheckSource(source);
        if (result.Defined()) {
            founded = true;
            output->Write(TPathResult{
                .Path = source.Path().GetPath(),
                .SearcherResults = result.GetRef(),
            });
        }
        output->Finish();
        return founded;
    }

    bool TSearcherStream::CheckStruct(NJson::TJsonValue& root) {
        bool founded = false;
        output->Start();
        auto results = worker->CheckStruct(root);
        if (!results.empty()) {
            founded = true;
            for (auto&& res : results) {
                // TODO(buglloc): WTF?!
                output->Write(TPathResult{
                    .Path = res.Path,
                    .SearcherResults = res.SearcherResults,
                });
            }
        }
        output->Finish();
        return founded;
    }

    bool TSearcherStream::CheckPath(const TFsPath& path) {
        NSSInternal::NFileWalker::TWalkerOptions walkerOpts{
            .Excludes = opts.Excludes};

        NSSInternal::NFileWalker::TWalker walker(walkerOpts);

        bool founded = false;
        auto walkFn = [this, &founded](const TString& path) -> bool {
            auto result = worker->CheckPath(path);
            if (result.Defined()) {
                founded = true;
                output->Write(result.GetRef());
            }
            return true;
        };

        output->Start();
        walker.Walk(path, walkFn);
        output->Finish();
        return founded;
    }

    TSearcherStreamThreaded::TSearcherStreamThreaded(const TSearchOptions& opts, NOutput::IWriter* writer,
                                                     size_t numThreads)
        : opts(opts)
        , ctx(opts.InternalCtx())
        , output(writer)
        , numThreads(numThreads)
    {
        Y_ENSURE(numThreads > 0, "To use not threaded searcher use simple TSearcherStream instead");
        for (size_t i = 0; i < numThreads; ++i) {
            workers.emplace_front(ctx);
        }
    }

    bool TSearcherStreamThreaded::CheckSource(ISource& source) {
        bool founded = false;
        output.GetWriter()->Start();
        auto result = workers[0].CheckSource(source);
        if (result.Defined()) {
            founded = true;
            output.GetWriter()->Write(TPathResult{
                .Path = source.Path().GetPath(),
                .SearcherResults = result.GetRef(),
            });
        }
        output.GetWriter()->Finish();
        return founded;
    }

    bool TSearcherStreamThreaded::CheckStruct(NJson::TJsonValue& root) {
        bool founded = false;
        output.GetWriter()->Start();
        auto results = workers[0].CheckStruct(root);
        if (!results.empty()) {
            founded = true;
            for (auto&& res : results) {
                // TODO(buglloc): WTF?!
                output.GetWriter()->Write(TPathResult{
                    .Path = res.Path,
                    .SearcherResults = res.SearcherResults,
                });
            }
        }
        output.GetWriter()->Finish();
        return founded;
    }

    bool TSearcherStreamThreaded::CheckPath(const TFsPath& path) {
        NSSInternal::TJobsQueue jobsQueue(kQueueSize);
        NSSInternal::TPathResultQueue resultsQueue(0);
        output.Start(&resultsQueue);

        auto* threadPool = SystemThreadFactory();
        TDeque<TAutoPtr<IThreadFactory::IThread>> threads;
        for (auto&& worker : workers) {
            worker.ResetQueue(&jobsQueue, &resultsQueue);
            threads.emplace_front(threadPool->Run(&worker));
        }

        NSSInternal::NFileWalker::TWalkerOptions walkerOpts{
            .Excludes = opts.Excludes};

        auto walkFn = [&jobsQueue](const TString& path) -> bool {
            jobsQueue.Push(path);
            return true;
        };

        NSSInternal::NFileWalker::TWalker walker(walkerOpts);
        walker.Walk(path, walkFn);

        // Wait workers
        jobsQueue.Stop();
        for (auto& thread : threads) {
            thread->Join();
        }

        output.Finish();
        return !output.Empty();
    }

}
