#include <security/ant-secret/secret-search/public/cpp/searcher_stream.h>
#include <security/ant-secret/secret-search/public/cpp/output/all.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/stream/str.h>
#include <util/folder/tempdir.h>
#include <util/folder/dirut.h>

Y_UNIT_TEST_SUITE(TSearcherStreamTests) {
    using namespace NSecretSearch;

    namespace {
        void writeFile(const TFsPath& path, const TString& content) {
            TFileOutput out(path);
            out.Write(content);
            out.Flush();
        }

        class IWriterFactory {
        public:
            virtual ~IWriterFactory() = default;

            virtual NOutput::IWriter* Create(IOutputStream& out) = 0;

            virtual const TString Name() const = 0;

            virtual const TString ExpectedEmpty() const = 0;
        };

        template <typename Writer>
        class TWriterFactory: public IWriterFactory {
        public:
            explicit TWriterFactory(TString name, TString emptyContent = "")
                : name(std::move(name))
                , emptyContent(std::move(emptyContent))
            {
            }

            ~TWriterFactory() override = default;

            NOutput::IWriter* Create(IOutputStream& out) override {
                return new Writer(out);
            }

            const TString Name() const override {
                return name;
            }

            const TString ExpectedEmpty() const override {
                return emptyContent;
            }

        protected:
            TString name;
            TString emptyContent;
        };

        TVector<TAutoPtr<IWriterFactory>> writers = {
            new TWriterFactory<NOutput::TText>("text"),
            new TWriterFactory<NOutput::TConsole>("console"),
            new TWriterFactory<NOutput::TStartrek>("startrek"),
            new TWriterFactory<NOutput::THector>("hector", "[]"),
            new TWriterFactory<NOutput::TJson>("json", "[]"),
            new TWriterFactory<NOutput::TJsonLines>("json lines"),
        };

        const TString kNotSecretContent = "something";

        const TString kSecretContent = R"(
telegram_token = '329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY'
oauth = '03c7c0ace395d80182db07ae2c30f034'
)";

    }

    Y_UNIT_TEST(TestContent) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        auto test = [opts](TSourceContent& source, bool secrets, IWriterFactory* writer) {
            TStringStream out;
            TSearcherStream searcher(opts, writer->Create(out));
            if (secrets) {
                if (!searcher.CheckSource(source)) {
                    ythrow yexception()
                        << "searcher can't find secrets in content with secrets, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Empty()) {
                    ythrow yexception()
                        << "searcher silently check secret content, writer: " << writer->Name()
                        << Endl;
                }
            } else {
                if (searcher.CheckSource(source)) {
                    ythrow yexception()
                        << "searcher FOUND secrets in NON secret content, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Str() != writer->ExpectedEmpty()) {
                    ythrow yexception()
                        << "searcher write '" << out.Str() << "' for NON secret content, writer: " << writer->Name()
                        << Endl;
                }
            }
        };

        for (const auto& writer : writers) {
            TSourceContent notSecretContent(kNotSecretContent);
            TSourceContent secretContent(kSecretContent);

            test(notSecretContent, false, writer.Get());
            test(secretContent, true, writer.Get());
        }
    }

    Y_UNIT_TEST(TestContentThreaded) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        auto test = [opts](TSourceContent& source, bool secrets, IWriterFactory* writer) {
            TStringStream out;
            TSearcherStreamThreaded searcher(opts, writer->Create(out), 2);
            if (secrets) {
                if (!searcher.CheckSource(source)) {
                    ythrow yexception()
                        << "searcher can't find secrets in content with secrets, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Empty()) {
                    ythrow yexception()
                        << "searcher silently check secret content, writer: " << writer->Name()
                        << Endl;
                }
            } else {
                if (searcher.CheckSource(source)) {
                    ythrow yexception()
                        << "searcher FOUND secrets in NON secret content, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Str() != writer->ExpectedEmpty()) {
                    ythrow yexception()
                        << "searcher write '" << out.Str() << "' for NON secret content, writer: " << writer->Name()
                        << Endl;
                }
            }
        };

        for (const auto& writer : writers) {
            TSourceContent notSecretContent(kNotSecretContent);
            TSourceContent secretContent(kSecretContent);

            test(notSecretContent, false, writer.Get());
            test(secretContent, true, writer.Get());
        }
    }

    Y_UNIT_TEST(TestPathSimple) {
        TTempDir tempDir("nonexistingdir");
        const auto& notSecretPath = tempDir.Path() / "not_secret";
        writeFile(notSecretPath, kNotSecretContent);
        const auto& secretPath = tempDir.Path() / "secret";
        writeFile(secretPath, kSecretContent);

        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        auto test = [opts](const TString& path, bool secrets, IWriterFactory* writer) {
            TStringStream out;
            TSearcherStream searcher(opts, writer->Create(out));
            if (secrets) {
                if (!searcher.CheckPath(path)) {
                    ythrow yexception()
                        << "searcher can't find secrets in path with secrets, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Empty()) {
                    ythrow yexception()
                        << "searcher silently check secret path, writer: " << writer->Name()
                        << Endl;
                }
            } else {
                if (searcher.CheckPath(path)) {
                    ythrow yexception()
                        << "searcher FOUND secrets in NON secret path, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Str() != writer->ExpectedEmpty()) {
                    ythrow yexception()
                        << "searcher write '" << out.Str() << "' for NON secret path, writer: " << writer->Name()
                        << Endl;
                }
            }
        };

        for (const auto& writer : writers) {
            TSourceContent notSecretContent(kNotSecretContent);
            TSourceContent secretContent(kSecretContent);

            test(notSecretPath, false, writer.Get());
            test(secretPath, true, writer.Get());
        }
    }

    Y_UNIT_TEST(TestPathThreaded) {
        TTempDir tempDir("nonexistingdir");
        const auto& notSecretPath = tempDir.Path() / "not_secret";
        writeFile(notSecretPath, kNotSecretContent);
        const auto& secretPath = tempDir.Path() / "secret";
        writeFile(secretPath, kSecretContent);

        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        auto test = [opts](const TString& path, bool secrets, IWriterFactory* writer) {
            TStringStream out;
            TSearcherStreamThreaded searcher(opts, writer->Create(out), 2);
            if (secrets) {
                if (!searcher.CheckPath(path)) {
                    ythrow yexception()
                        << "searcher can't find secrets in path with secrets, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Empty()) {
                    ythrow yexception()
                        << "searcher silently check secret path, writer: " << writer->Name()
                        << Endl;
                }
            } else {
                if (searcher.CheckPath(path)) {
                    ythrow yexception()
                        << "searcher FOUND secrets in NON secret path, writer: " << writer->Name()
                        << Endl;
                }
                out.Flush();
                if (out.Str() != writer->ExpectedEmpty()) {
                    ythrow yexception()
                        << "searcher write '" << out.Str() << "' for NON secret path, writer: " << writer->Name()
                        << Endl;
                }
            }
        };

        for (const auto& writer : writers) {
            TSourceContent notSecretContent(kNotSecretContent);
            TSourceContent secretContent(kSecretContent);

            test(notSecretPath, false, writer.Get());
            test(secretPath, true, writer.Get());
        }
    }

    Y_UNIT_TEST(TestNotExisted) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TStringStream out;
        TSearcherStream simpleSearcher(opts, new NOutput::TText(out));
        TSearcherStreamThreaded threadedSearcher(opts, new NOutput::TText(out), 2);

        UNIT_ASSERT(!simpleSearcher.CheckPath("not_existed"));
        UNIT_ASSERT(!threadedSearcher.CheckPath("not_existed"));
    }
}
