#include <security/ant-secret/secret-search/public/cpp/searcher.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/stream/str.h>
#include <util/folder/tempdir.h>
#include <util/folder/dirut.h>

Y_UNIT_TEST_SUITE(TSearcherTests) {
    using namespace NSecretSearch;

    namespace {
        void writeFile(const TFsPath& path, const TString& content) {
            TFileOutput out(path);
            out.Write(content);
            out.Flush();
        }

    }

    Y_UNIT_TEST(TestContentSimple) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcher searcher(opts);
        TSourceContent notSecretContent("something");
        TSourceContent secretContent(R"(
telegram_token = '329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY'
oauth = '03c7c0ace395d80182db07ae2c30f034'
iam_token = 't1.9f7P7fy2m87l9_dXQSMT-u_-3ff3F3AgE_rv_tXu9fickJGMkJOa7fmQj5qRlps=.tMiX952V4Q3T6c5RSBMv6TeZ5DPQHT4hbBR9EgUNvApjK_gi8OMEkzekju512LkR6figaq-P5fI2_i82pBx8Bg=='
yc_session = 'c1.9f7P7fy2m87l9_dXQSMT-u_X3ff3F3AgE_rv19Xp9fOQnoqLl6CMmo2Jmo3t-ZCPmpGWmw==.1jve0Jdy_ZhiQLuWw5A4fZXtaaJXJgdIgLJdd_3JRA79sa1pI9bgZo8-zUx2JQWB9Uf8bb95bNfnReihX6LpDw=='
yc_api_key = 'AQVN2dKlgqLGcHhM7wPrpCTfr5m8tqNn8sz_-i_t'
yc_secret: 'YCOFcu7iDT8zC-p0fGkOfEGaaWzHY6TFe3t3av43'
)");
        UNIT_ASSERT(!searcher.CheckSource(notSecretContent).Defined());
        auto result = searcher.CheckSource(secretContent);
        UNIT_ASSERT(result.Defined());
        auto tokens = result.GetRef();
        UNIT_ASSERT_EQUAL(tokens.size(), 6);

        auto token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 7);
        UNIT_ASSERT_EQUAL(token.Secret, "YCOFcu7iDT8zC-p0fGkOfEGaaWzHY6TFe3t3av43");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "ae4d3052981752d51af8e6ca062a84c344039bc7");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 6);
        UNIT_ASSERT_EQUAL(token.Secret, "AQVN2dKlgqLGcHhM7wPrpCTfr5m8tqNn8sz_-i_t");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "915c79cd7447073dc77e9d376c68171ed54081ee");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 5);
        UNIT_ASSERT_EQUAL(token.Secret, "c1.9f7P7fy2m87l9_dXQSMT-u_X3ff3F3AgE_rv19Xp9fOQnoqLl6CMmo2Jmo3t-ZCPmpGWmw==.1jve0Jdy_ZhiQLuWw5A4fZXtaaJXJgdIgLJdd_3JRA79sa1pI9bgZo8-zUx2JQWB9Uf8bb95bNfnReihX6LpDw==");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "dd22afb8ae51d22c8abc7499ca5d27aac6782482");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 4);
        UNIT_ASSERT_EQUAL(token.Secret, "t1.9f7P7fy2m87l9_dXQSMT-u_-3ff3F3AgE_rv_tXu9fickJGMkJOa7fmQj5qRlps=.tMiX952V4Q3T6c5RSBMv6TeZ5DPQHT4hbBR9EgUNvApjK_gi8OMEkzekju512LkR6figaq-P5fI2_i82pBx8Bg==");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "4c2def369c3f4db02e1e8db265cecf7d3be0062d");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 3);
        UNIT_ASSERT_EQUAL(token.Secret, "03c7c0ace395d80182db07ae2c30f034");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "d860be0ab4a0189ef35a449cd100ef6e9c0d983f");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 2);
        UNIT_ASSERT_EQUAL(token.Secret, "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "6d83cf93f702f19a0a453dc0b0323f7969f92e4d");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);
    }

    Y_UNIT_TEST(TestContentThreaded) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcherThreaded searcher(opts, 2);
        TSourceContent notSecretContent("something");
        TSourceContent secretContent(R"(
telegram_token = '329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY'
oauth = '03c7c0ace395d80182db07ae2c30f034'
)");
        UNIT_ASSERT(!searcher.CheckSource(notSecretContent).Defined());
        auto result = searcher.CheckSource(secretContent);
        UNIT_ASSERT(result.Defined());
        auto tokens = result.GetRef();
        UNIT_ASSERT_EQUAL(tokens.size(), 2);

        auto token = tokens.back();

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 3);
        UNIT_ASSERT_EQUAL(token.Secret, "03c7c0ace395d80182db07ae2c30f034");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "d860be0ab4a0189ef35a449cd100ef6e9c0d983f");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 2);
        UNIT_ASSERT_EQUAL(token.Secret, "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "6d83cf93f702f19a0a453dc0b0323f7969f92e4d");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);
    }

    Y_UNIT_TEST(TestStructSimple) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcher searcher(opts);
        NJson::TJsonValue notSecretJson;
        NJson::ReadJsonTree(R"(["not", "a", "secret"])", &notSecretJson, true);

        NJson::TJsonValue secretJson;
        NJson::ReadJsonTree(R"({"telegram_token": "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY", "oauth": ["cheburek", "03c7c0ace395d80182db07ae2c30f034"]})", &secretJson, true);

        UNIT_ASSERT(searcher.CheckStruct(notSecretJson).empty());
        auto results = searcher.CheckStruct(secretJson);
        UNIT_ASSERT_EQUAL(results.size(), 2);

        auto secret = results.back();
        results.pop_back();
        UNIT_ASSERT_STRINGS_EQUAL(secret.Path, "$.telegram_token");
        UNIT_ASSERT_EQUAL(secret.SearcherResults.size(), 1);
        UNIT_ASSERT_STRINGS_EQUAL(secret.SearcherResults[0].Secret, "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY");
        UNIT_ASSERT(!secret.SearcherResults[0].Additional.empty());
        UNIT_ASSERT_STRINGS_EQUAL(secret.SearcherResults[0].Additional["sha1"], "6d83cf93f702f19a0a453dc0b0323f7969f92e4d");
        UNIT_ASSERT(!secret.SearcherResults[0].Validated);
        UNIT_ASSERT(!secret.SearcherResults[0].Ignore);

        secret = results.back();
        results.pop_back();
        UNIT_ASSERT_STRINGS_EQUAL(secret.Path, "$.oauth[1]");
        UNIT_ASSERT_EQUAL(secret.SearcherResults.size(), 1);
        UNIT_ASSERT_STRINGS_EQUAL(secret.SearcherResults[0].Secret, "03c7c0ace395d80182db07ae2c30f034");
        UNIT_ASSERT(!secret.SearcherResults[0].Additional.empty());
        UNIT_ASSERT_STRINGS_EQUAL(secret.SearcherResults[0].Additional["sha1"], "d860be0ab4a0189ef35a449cd100ef6e9c0d983f");
        UNIT_ASSERT(!secret.SearcherResults[0].Validated);
        UNIT_ASSERT(!secret.SearcherResults[0].Ignore);
    }

    Y_UNIT_TEST(TestStructThreaded) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcherThreaded searcher(opts, 2);
        NJson::TJsonValue notSecretJson;
        NJson::ReadJsonTree(R"(["not", "a", "secret"])", &notSecretJson, true);

        NJson::TJsonValue secretJson;
        NJson::ReadJsonTree(R"({"telegram_token": "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY", "oauth": ["cheburek", "03c7c0ace395d80182db07ae2c30f034"]})", &secretJson, true);

        UNIT_ASSERT(searcher.CheckStruct(notSecretJson).empty());
        auto results = searcher.CheckStruct(secretJson);
        UNIT_ASSERT_EQUAL(results.size(), 2);

        auto secret = results.back();
        results.pop_back();
        UNIT_ASSERT_EQUAL(secret.Path, "$.telegram_token");
        UNIT_ASSERT_EQUAL(secret.SearcherResults.size(), 1);
        UNIT_ASSERT_EQUAL(secret.SearcherResults[0].Secret, "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY");
        UNIT_ASSERT(!secret.SearcherResults[0].Additional.empty());
        UNIT_ASSERT_EQUAL(secret.SearcherResults[0].Additional["sha1"], "6d83cf93f702f19a0a453dc0b0323f7969f92e4d");
        UNIT_ASSERT(!secret.SearcherResults[0].Validated);
        UNIT_ASSERT(!secret.SearcherResults[0].Ignore);

        secret = results.back();
        results.pop_back();
        UNIT_ASSERT_EQUAL(secret.Path, "$.oauth[1]");
        UNIT_ASSERT_EQUAL(secret.SearcherResults.size(), 1);
        UNIT_ASSERT_EQUAL(secret.SearcherResults[0].Secret, "03c7c0ace395d80182db07ae2c30f034");
        UNIT_ASSERT(!secret.SearcherResults[0].Additional.empty());
        UNIT_ASSERT_EQUAL(secret.SearcherResults[0].Additional["sha1"], "d860be0ab4a0189ef35a449cd100ef6e9c0d983f");
        UNIT_ASSERT(!secret.SearcherResults[0].Validated);
        UNIT_ASSERT(!secret.SearcherResults[0].Ignore);
    }

    Y_UNIT_TEST(TestPathSimple) {
        TTempDir tempDir("nonexistingdir");
        const auto& notSecretPath = tempDir.Path() / "not_secret";
        writeFile(notSecretPath, "not a secret");
        const auto& secretPath = tempDir.Path() / "secret";
        writeFile(secretPath, R"(
telegram_token = '329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY'
oauth = '03c7c0ace395d80182db07ae2c30f034'
)");

        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcher searcher(opts);
        UNIT_ASSERT(searcher.CheckPath(notSecretPath).empty());
        auto result = searcher.CheckPath(secretPath);
        UNIT_ASSERT(!result.empty());
        UNIT_ASSERT_EQUAL(result[0].Path, secretPath.GetPath());
        auto tokens = result[0].SearcherResults;
        UNIT_ASSERT_EQUAL(tokens.size(), 2);

        auto token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 3);
        UNIT_ASSERT_EQUAL(token.Secret, "03c7c0ace395d80182db07ae2c30f034");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "d860be0ab4a0189ef35a449cd100ef6e9c0d983f");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 2);
        UNIT_ASSERT_EQUAL(token.Secret, "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "6d83cf93f702f19a0a453dc0b0323f7969f92e4d");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);
    }

    Y_UNIT_TEST(TestPathThreaded) {
        TTempDir tempDir("nonexistingdir");
        const auto& notSecretPath = tempDir.Path() / "not_secret";
        writeFile(notSecretPath, "not a secret");
        const auto& secretPath = tempDir.Path() / "secret";
        writeFile(secretPath, R"(
telegram_token = '329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY'
oauth = '03c7c0ace395d80182db07ae2c30f034'
)");

        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcherThreaded searcher(opts, 2);
        UNIT_ASSERT(searcher.CheckPath(notSecretPath).empty());
        auto result = searcher.CheckPath(secretPath);
        UNIT_ASSERT(!result.empty());
        UNIT_ASSERT_EQUAL(result[0].Path, secretPath.GetPath());
        auto tokens = result[0].SearcherResults;
        UNIT_ASSERT_EQUAL(tokens.size(), 2);

        auto token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 3);
        UNIT_ASSERT_EQUAL(token.Secret, "03c7c0ace395d80182db07ae2c30f034");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "d860be0ab4a0189ef35a449cd100ef6e9c0d983f");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);

        token = tokens.back();
        tokens.pop_back();
        UNIT_ASSERT_EQUAL(token.LineNo, 2);
        UNIT_ASSERT_EQUAL(token.Secret, "329142621:AAEYEkfcQ8wVpPFeQCY8IRY2M7dUVQ6ULuY");
        UNIT_ASSERT(!token.Additional.empty());
        UNIT_ASSERT_EQUAL(token.Additional["sha1"], "6d83cf93f702f19a0a453dc0b0323f7969f92e4d");
        UNIT_ASSERT(!token.Validated);
        UNIT_ASSERT(!token.Ignore);
    }

    Y_UNIT_TEST(TestNotExisted) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcher simpleSearcher(opts);
        TSearcherThreaded threadedSearcher(opts, 2);

        UNIT_ASSERT(simpleSearcher.CheckPath("not_existed").empty());
        UNIT_ASSERT(threadedSearcher.CheckPath("not_existed").empty());
    }

    Y_UNIT_TEST(TestRequestIDIgnored) {
        TSearchOptions opts{
            .Validate = false,
            .ValidOnly = false,
        };

        TSearcher searcher(opts);
        TSourceContent secretContent(R"(
request_id=03c7c0ace395d80182db07ae2c30f034 // auth
)");
        auto result = searcher.CheckSource(secretContent);
        UNIT_ASSERT(!result.Defined() || result->empty());
    }

    Y_UNIT_TEST(TestOAuth) {
        TSearchOptions opts{
                .Validate = false,
                .ValidOnly = false,
        };

        TSearcher searcher(opts);
        TSourceContent secretContent(R"(
token = "AQAD-qJSJpcjAAADwHXXXXXXXXXXXXXXXXXXXXX"
token = "Attribution-NonCommercial-NoDerivatives"
token = "AQAAAAAN_UPlAAIbup_7VdW7UERdvcRTC1D4Ee0A"
)");
        auto result = searcher.CheckSource(secretContent);
        UNIT_ASSERT(!result.Defined() || result->empty());
    }
}
