import six

from cpython cimport PyBytes_AS_STRING
from util.generic.string cimport TString
from util.generic.hash cimport THashMap
from util.generic.vector cimport TVector
from util.generic.maybe cimport TMaybe
from libcpp cimport bool


cdef extern from "security/ant-secret/secret-search/public/cpp/secret.h" namespace "NSecretSearch":
    ctypedef THashMap[TString, TString] TSecretAdditional

    cdef cppclass TSecret:
        TString Type
        TString Secret
        TString SourceLine
        size_t LineNo
        THashMap[TString, TString] Additional
        bool Ignore
        bool Validated

    ctypedef TVector[TSecret] TSecretList


cdef extern from "security/ant-secret/secret-search/public/cpp/search_result.h" namespace "NSecretSearch":
    ctypedef TSecretList TSourceResult

    cdef cppclass TPathResult:
        TString Path
        TSourceResult SearcherResults

    cdef cppclass TStructResult:
        TString Path
        TSourceResult SearcherResults


cdef extern from "security/ant-secret/secret-search/public/cpp/search_options.h" namespace "NSecretSearch":
    cdef cppclass TSearchOptions:
        size_t MaxFileSize
        TVector[TString] Excludes
        bool Validate
        bool ValidOnly


cdef extern from "security/ant-secret/secret-search/public/py/simple.h" namespace "NSecretSearch":
    cdef cppclass TSimpleSearcher:
        TSimpleSearcher(TSearchOptions)
        TMaybe[TSourceResult] CheckSource(const TString& content)
        TVector[TPathResult] CheckPath(const TString& path)
        TVector[TStructResult] CheckJson(const TString& content)


class Secret(object):
    __slots__ = ('type', 'secret', 'source_line', 'line_no', 'validated', 'additional')

    def __init__(self, typ, secret, source_line, line_no, validated, additional):
        self.type = six.ensure_text(typ)
        self.secret = six.ensure_text(secret)
        self.source_line = six.ensure_text(source_line)
        self.line_no = line_no
        self.validated = validated
        self.additional = {six.ensure_text(k): six.ensure_text(v) for k, v in additional.items()}

cdef class Searcher(object):
    cdef TSimpleSearcher* searcher

    def __cinit__(self, bool validate=False, bool valid_only=False, TVector[TString] excludes=[], size_t max_file_size=3<<20):
        cdef TSearchOptions opts
        opts.MaxFileSize = max_file_size
        opts.Validate = validate
        opts.ValidOnly = valid_only
        opts.Excludes = excludes
        self.searcher = new TSimpleSearcher(opts)

    def __dealloc__(self):
        if self.searcher is not NULL:
            del self.searcher
            self.searcher = NULL

    def check_blob(self, content):
        source = TString(PyBytes_AS_STRING(six.ensure_binary(content)))
        maybe_result = self.searcher.CheckSource(source)
        if not maybe_result.Defined():
            return []

        for secret in maybe_result.GetRef():
            yield Secret(
                typ=secret.Type,
                secret=secret.Secret,
                source_line=secret.SourceLine,
                line_no=secret.LineNo,
                validated=secret.Validated,
                additional=secret.Additional,
            )

    def check_path(self, path):
        source_path = TString(PyBytes_AS_STRING(six.ensure_binary(path)))
        path_results = self.searcher.CheckPath(source_path)
        for path_result in path_results:
            secrets = []
            for secret in path_result.SearcherResults:
                secrets.append(Secret(
                    typ=secret.Type,
                    secret=secret.Secret,
                    source_line=secret.SourceLine,
                    line_no=secret.LineNo,
                    validated=secret.Validated,
                    additional=secret.Additional,
                ))
            yield six.ensure_text(path_result.Path), secrets

    def check_json(self, content):
        source = TString(PyBytes_AS_STRING(six.ensure_binary(content)))
        json_results = self.searcher.CheckJson(source)
        for json_result in json_results:
            secrets = []
            for secret in json_result.SearcherResults:
                secrets.append(Secret(
                    typ=secret.Type,
                    secret=secret.Secret,
                    source_line=secret.SourceLine,
                    line_no=secret.LineNo,
                    validated=secret.Validated,
                    additional=secret.Additional,
                ))
            yield six.ensure_text(json_result.Path), secrets
