#include "options.h"

#include <library/cpp/getopt/last_getopt.h>
#include <util/string/vector.h>
#include <util/string/strip.h>
#include <util/string/join.h>
#include <util/system/env.h>

namespace NSecretSearchServer {
    const TProgramOptions TProgramOptions::Parse(int argc, char** argv) {
        auto options = TProgramOptions();

        using namespace NLastGetopt;
        TOpts opts = TOpts::Default();

        opts.SetTitle("SecretSearchServer server");

        opts.AddLongOption("addr", "server addr to bind")
            .RequiredArgument("HOST:PORT")
            .DefaultValue("127.0.0.1:5000")
            .StoreResult(&options.ServerAddr);

        opts.AddLongOption("no-validation", "disable validation")
            .NoArgument();

        opts.AddLongOption("tvm-clients", "comma-separated list of allowed TVM client ids")
            .Optional();

        TOptsParseResult args(&opts, argc, argv);
        if (args.Has("tvm-clients")) {
            for (const auto& id : SplitString(args.Get("tvm-clients"), ",")) {
                options.AllowedTvms.push_back(FromString<size_t>(StripString(id)));
            }
        }

        options.TvmToolToken = GetEnv("TVMTOOL_LOCAL_AUTHTOKEN");
        options.ValidateSecrets = !args.Has("no-validation");
        return options;
    }

    void TProgramOptions::Print(IOutputStream& out) {
        out
            << "ServerAddr = " << ServerAddr << Endl
            << "TVM clients = " << JoinSeq(",", AllowedTvms) << Endl
            << "TVM tool auth token = " << (TvmToolToken ? "*****" : "") << Endl;
    }

    const TServerOptions TProgramOptions::ServerOptions() const {
        return TServerOptions{
            .ValidateSecrets = ValidateSecrets,
            .ServerAddr = ServerAddr,
            .TvmToolToken = TvmToolToken,
            .AllowedTvms = AllowedTvms,
        };
    }

}
