#include "diff.h"

#include <util/generic/string.h>
#include <util/generic/strbuf.h>
#include <util/generic/maybe.h>
#include <util/string/vector.h>
#include <util/string/join.h>

namespace NSecretSearchServer {
    namespace {
        enum EState {
            Begin,
            SourceFile,
            TargetFile,
            HunkHeader,
            End,
        };

        struct THunkHeader {
            size_t SourcePos = 0;
            size_t TargetPos = 0;
            size_t SourceSize = 0;
            size_t TargetSize = 0;
        };

        THunkHeader ParseHunkHeader(const TStringBuf line) {
            auto parts = SplitString(TString{line}, " ");
            if (parts.size() != 4) {
                ythrow TDiffParseException() << "unexpected hunk header: " << line;
            }

            auto from = parts[1];
            if (from.empty() || from[0] != '-') {
                ythrow TDiffParseException() << "unexpected from-file in hunk header: " << from;
            }
            from = from.substr(1);

            auto to = parts[2];
            if (to.empty() || to[0] != '+') {
                ythrow TDiffParseException() << "unexpected to-file in hunk header: " << from;
            }
            to = to.substr(1);

            THunkHeader result;
            auto splittedFrom = SplitString(from, ",");
            if (splittedFrom.empty() || splittedFrom.size() > 2) {
                ythrow TDiffParseException() << "unexpected from-file in hunk header: " << from;
            }

            if (splittedFrom.size() == 1) {
                // If a hunk contains just one line, only its start line number appears.
                // Otherwise its line numbers look like ‘start,count’.
                result.SourcePos = 0;
                result.SourceSize = FromString<size_t>(from);
            } else {
                result.SourcePos = FromString<size_t>(splittedFrom[0]);
                if (result.SourcePos) {
                    result.SourcePos--;
                }
                result.SourceSize = FromString<size_t>(splittedFrom[1]);
            }

            auto splittedTo = SplitString(to, ",");
            if (splittedTo.empty() || splittedTo.size() > 2) {
                ythrow TDiffParseException() << "unexpected to-file in hunk header: " << to;
            }

            if (splittedTo.size() == 1) {
                // If a hunk contains just one line, only its start line number appears.
                // Otherwise its line numbers look like ‘start,count’.
                result.TargetPos = 0;
                result.TargetSize = FromString<size_t>(to);
            } else {
                result.TargetPos = FromString<size_t>(splittedTo[0]);
                if (result.TargetPos) {
                    result.TargetPos--;
                }
                result.TargetSize = FromString<size_t>(splittedTo[1]);
            }

            return result;
        }

        void ParseHunkChunk(TDiff& diff, TStringBuf& source, const THunkHeader& header) {
            size_t removedPos = header.SourcePos;
            size_t expectedRemovedPos = header.SourcePos + header.SourceSize;
            size_t addedPos = header.TargetPos;
            size_t expectedAddedPos = header.TargetPos + header.TargetSize;
            TStringBuf line = nullptr;
            bool tryNext = true;
            while (tryNext && source.ReadLine(line)) {
                if (line.empty()) {
                    removedPos++;
                    addedPos++;
                    continue;
                }

                switch (line[0]) {
                    case ' ':
                        removedPos++;
                        addedPos++;
                        break;
                    case '-':
                        removedPos++;
                        diff.Removed.push_back(TDiffLine{
                            .LineNo = removedPos,
                            .Line = line.Skip(1)});
                        break;
                    case '+':
                        addedPos++;
                        diff.Added.push_back(TDiffLine{
                            .LineNo = addedPos,
                            .Line = line.Skip(1)});
                        break;
                    case '\\':
                        tryNext = false;
                        continue;

                    default:
                        ythrow TDiffParseException() << "unexpected chunk line: " << line;
                }

                tryNext = removedPos < expectedRemovedPos || addedPos < expectedAddedPos;
            }
        }

    }

    /*
https://www.gnu.org/software/diffutils/manual/html_node/Unified-Format.html
--- /dev/null 2018-11-08 14:36:09.883332277 +0300
+++ 123 2018-11-09 12:41:07.102448420 +0300
@@ -0,0 +1 @@
+2
 */
    TPatchSet ParsePatchSet(const TString& source) {
        TPatchSet result;
        TStringBuf line = nullptr;
        TMaybe<TDiff> diff;
        TStringBuf content(source);
        EState state = EState::Begin;
        bool tryNext = true;
        while (tryNext) {
            switch (state) {
                case EState::Begin:
                    state = EState::SourceFile;
                    break;
                case EState::SourceFile:
                    if (line.StartsWith(TStringBuf("--- "))) {
                        // What about space in file names ? :(
                        diff = TDiff();
                        auto delim = line.find_first_of(" \t", 4);
                        if (delim != NPOS) {
                            diff->SourceFile = line.SubStr(4, delim - 4);
                        } else {
                            diff->SourceFile = line.SubStr(4);
                        }
                        state = EState::TargetFile;
                    }
                    break;
                case EState::TargetFile:
                    if (Y_LIKELY(line.StartsWith(TStringBuf("+++ ")))) {
                        // What about space in file names ? :(
                        auto delim = line.find_first_of(" \t", 4);
                        if (delim != NPOS) {
                            diff->TargetFile = line.substr(4, delim - 4);
                        } else {
                            diff->TargetFile = line.substr(4);
                        }
                        state = EState::HunkHeader;
                    }
                    break;
                case EState::HunkHeader:
                    if (Y_LIKELY(line.StartsWith(TStringBuf("@@ ")))) {
                        auto header = ParseHunkHeader(line);
                        ParseHunkChunk(diff.GetRef(), content, header);
                        tryNext = content.ReadLine(line);
                        if (tryNext) {
                            if (!line.StartsWith(TStringBuf("@@ "))) {
                                // if we checked las hunk
                                state = EState::End;
                            }
                            continue;
                        }
                    }
                    break;
                case EState::End:
                    if (diff.Defined()) {
                        result.push_back(diff.GetRef());
                        diff.Clear();
                    }
                    state = EState::SourceFile;
                    break;
            }

            tryNext = content.ReadLine(line);
        }

        if (diff.Defined() && (!diff->Removed.empty() || !diff->Added.empty())) {
            result.push_back(diff.GetRef());
            diff.Clear();
        }
        return result;
    }

}
