#include <security/libs/cpp/log/log.h>
#include "searcher.h"
#include "diff_source.h"

namespace NSecretSearchServer {
    namespace {
        constexpr size_t kQueueSize = 1024;

        const TString kApiUrl = "http://localhost:80/api/v1/validate/";

    }

    TSearcher::TSearcher(const TSearchOptions& opts, size_t numThreads)
        : opts(opts)
        , ctx(opts.InternalCtx())
        , numThreads(numThreads)
    {
        Y_ENSURE(numThreads > 0, "num threads must be gather than 0");
        for (size_t i = 0; i < numThreads; ++i) {
            workers.emplace_front(ctx);
        }
    }

    TMaybe<NSSInternal::TSourceResult> TSearcher::CheckContent(const TString& input) {
        NSSInternal::TSourceContent source(input);
        ctx.Validator.Reset(new TValidator(kApiUrl));
        return workers[0].CheckSource(source);
    }

    TVector<NSSInternal::TPathResult> TSearcher::CheckPatchSet(const TString& input) {
        TJobsQueue jobsQueue(kQueueSize);
        NSSInternal::TPathResultQueue resultsQueue(0);

        auto* threadPool = SystemThreadFactory();
        TDeque<TAutoPtr<IThreadFactory::IThread>> threads;
        for (auto&& worker : workers) {
            worker.ResetState(&jobsQueue, &resultsQueue);
            threads.emplace_front(threadPool->Run(&worker));
        }

        auto pathSet = ParsePatchSet(input);
        ctx.Validator.Reset(new TValidator(kApiUrl, &pathSet));

        for (auto&& diff : pathSet) {
            TSourceDiff source(diff.TargetFile, diff.Added);
            jobsQueue.Push(source);
        }

        // Wait workers
        jobsQueue.Stop();
        for (auto& thread : threads) {
            thread->Join();
        }

        // Parse results
        resultsQueue.Stop();
        TVector<NSSInternal::TPathResult> results;
        results.reserve(resultsQueue.Size());
        while (true) {
            auto maybeResult = resultsQueue.Pop();
            if (maybeResult.Empty())
                break;

            results.push_back(maybeResult.GetRef());
        }
        return results;
    }

}
