#include "server.h"

#include <security/libs/cpp/log/log.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/neh/neh.h>
#include <library/cpp/neh/http_common.h>
#include <util/string/builder.h>
#include <util/generic/strbuf.h>
#include <util/string/strip.h>
#include <util/string/join.h>

namespace NSecretSearchServer {
    namespace {
        constexpr TStringBuf kPingPath = "/ping";

        constexpr TStringBuf kVcsHookPath = "/hook";

        constexpr TStringBuf kDiffPath = "/diff";

        constexpr TStringBuf kContentPath = "/content";

        const TString kPingOutput = "ok";

        const TString kTvmTicketHeader = "X-Ya-Service-Ticket: ";

        const TString kTvmAuthHeader = "Authorization: ";

        const TString kInputTvmTicketHeader = "X-Ya-Service-Ticket";

        const TString kTvmToolUrl = "http://localhost:2/tvm/checksrv?dst=self";

        constexpr TDuration kTvmDeadline = TDuration::Seconds(1);

        constexpr int kMaxConnections = 20;

        constexpr int kSearchThreads = 4;

    }

    TSearchReplier::TSearchReplier(const TVector<size_t>& allowedTvms, const TString& TvmToolToken, bool validateSecrets)
        : allowedTvms(allowedTvms)
        , tvmToolToken(TvmToolToken)
        , searcher(TSearchOptions{.Validate = validateSecrets, .ValidOnly = validateSecrets}, kSearchThreads)
    {
    }

    bool TSearchReplier::DoReply(const TReplyParams& params) {
        THttpResponse resp;
        try {
            TParsedHttpFull req(params.Input.FirstLine());

            if (req.Path == kPingPath) {
                resp.SetHttpCode(HTTP_OK);
                resp.SetContent(kPingOutput);
                params.Output << resp;
                return true;
            }

            if (req.Method != "POST" || !params.Input.HasContent()) {
                resp.SetHttpCode(HTTP_NOT_FOUND);
                resp.SetContent("Only POST w/ content allowed");
                params.Output << resp;
                return true;
            }

            if (!allowedTvms.empty()) {
                TString serviceTicket;
                for (const auto& header : params.Input.Headers()) {
                    if (header.Name() == kInputTvmTicketHeader) {
                        serviceTicket = header.Value();
                    }
                }

                if (!serviceTicket || !CheckAuth(serviceTicket)) {
                    resp.SetHttpCode(HTTP_UNAUTHORIZED);
                    resp.SetContent("TVM authorization required, acceptable clients: " + JoinSeq(",", allowedTvms));
                    params.Output << resp;
                    return true;
                }
            }

            TStringStream out;
            NJson::TJsonWriter jsonWriter(&out, /*formatOutput*/ false, /*sortkeys*/ false, /*validateUtf8*/ false);
            auto content = params.Input.ReadAll();
            if (req.Path == kVcsHookPath) {
                auto results = searcher.CheckPatchSet(content);
                jsonWriter.OpenArray();
                for (auto&& result : results) {
                    for (auto&& secret : result.SearcherResults) {
                        jsonWriter.OpenMap();
                        jsonWriter.Write("path", result.Path);
                        jsonWriter.Write("line", secret.LineNo);
                        jsonWriter.Write("type", secret.Type);
                        jsonWriter.Write("info", secret.ShortAdditional());
                        jsonWriter.CloseMap();
                    }
                }
                jsonWriter.CloseArray();
            } else if (req.Path == kDiffPath) {
                auto results = searcher.CheckPatchSet(content);
                jsonWriter.OpenArray();
                for (const auto& result : results) {
                    result.ToJson(jsonWriter);
                }
                jsonWriter.CloseArray();
            } else if (req.Path == kContentPath) {
                auto secrets = searcher.CheckContent(content);
                jsonWriter.OpenArray();
                if (secrets.Defined()) {
                    for (const auto& secret : secrets.GetRef()) {
                        jsonWriter.OpenMap();
                        jsonWriter.Write("line", secret.LineNo);
                        jsonWriter.Write("type", secret.Type);
                        jsonWriter.Write("info", secret.ShortAdditional());
                        jsonWriter.CloseMap();
                    }
                }
                jsonWriter.CloseArray();
            } else {
                resp.SetHttpCode(HTTP_NOT_FOUND);
                resp.SetContent("Path was not found, sorry :(");
                params.Output << resp;
                return true;
            }

            jsonWriter.Flush();
            resp.SetContent(out.Str());
        } catch (yexception& e) {
            Cerr << (TStringBuilder() << "REQUEST_ERROR: " << e.what() << Endl);
            resp.SetHttpCode(HTTP_BAD_REQUEST);
        }

        params.Output << resp;
        return true;
    }

    bool TSearchReplier::CheckAuth(const TString& serviceTicket) {
        auto req = NNeh::TMessage(kTvmToolUrl, nullptr);
        TStringBuilder headers;
        headers << kTvmTicketHeader << serviceTicket << "\r\n";
        if (!tvmToolToken.empty()) {
            headers << kTvmAuthHeader << tvmToolToken << "\r\n";
        }

        bool ok = NNeh::NHttp::MakeFullRequest(req, headers, TString(), TString(), NNeh::NHttp::ERequestType::Get);
        if (Y_UNLIKELY(!ok)) {
            NSecurityHelpers::LogErr("Failed to TvmTool request");
            return false;
        }

        NNeh::TResponseRef resp;
        if (!NNeh::Request(req)->Wait(resp, kTvmDeadline)) {
            NSecurityHelpers::LogErr("Failed to call TvmTool", "err", "timed out");
            return false;
        }

        if (Y_UNLIKELY(!resp)) {
            NSecurityHelpers::LogErr("Failed to call TvmTool", "err", "empty response");
            return false;
        }

        if (resp->IsError()) {
            NSecurityHelpers::LogErr("Failed to call TvmTool service", "err", resp->GetErrorText());
            return false;
        }

        NJson::TJsonValue data;
        if (!NJson::ReadJsonTree(resp->Data, &data)) {
            NSecurityHelpers::LogErr("Failed to parse TvmTool response", "response", resp->Data);
            return false;
        }

        auto clientId = data["src"].GetUIntegerSafe();
        return std::find(allowedTvms.begin(), allowedTvms.end(), clientId) != allowedTvms.end();
    }

    void StartServer(const TServerOptions& opts) {
        THttpServer::TOptions httpServerOptions;
        httpServerOptions.MaxConnections = kMaxConnections;
        httpServerOptions.KeepAliveEnabled = true;
        {
            size_t colon = opts.ServerAddr.rfind(':');
            httpServerOptions.Host = opts.ServerAddr.substr(0, colon);
            if (!TryFromString(opts.ServerAddr.substr(colon + 1), httpServerOptions.Port)) {
                httpServerOptions.Port = 10203;
            }
        }

        THttpCallback httpCallback(
            opts.AllowedTvms,
            opts.TvmToolToken,
            opts.ValidateSecrets
        );
        THttpServer httpServer(&httpCallback, httpServerOptions);
        httpServer.Start();
        Cerr << "Server started" << Endl;
        httpServer.Wait();
    }

}
