#include <security/ant-secret/secret-search/server/lib/diff.h>

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(TDiffTests) {
    using namespace NSecretSearchServer;

    Y_UNIT_TEST(Broken) {
        {
            TString source(R"(
--- /dev/null 2018-11-08 14:36:09.883332277 +0300
+++ 123 2018-11-09 12:41:07.102448420 +0300
@@ -0,0 @@
+2
)");
            UNIT_CHECK_GENERATED_EXCEPTION(ParsePatchSet(source), TDiffParseException);
        }

        {
            TString source(R"(
--- /dev/null 2018-11-08 14:36:09.883332277 +0300
+++ 123 2018-11-09 12:41:07.102448420 +0300
@@ -0,0 @@
+2
)");
            UNIT_CHECK_GENERATED_EXCEPTION(ParsePatchSet(source), TDiffParseException);
        }

        {
            TString source(R"(
--- /dev/null 2018-11-08 14:36:09.883332277 +0300
+++ 123 2018-11-09 12:41:07.102448420 +0300
@@ -0,0 +1 @@
2
)");
            UNIT_CHECK_GENERATED_EXCEPTION(ParsePatchSet(source), TDiffParseException);
        }

        {
            TString source(R"(
--- /dev/null 2018-11-08 14:36:09.883332277 +0300
+++ 123 2018-11-09 12:41:07.102448420 +0300
)");
            auto patchSet = ParsePatchSet(source);
            UNIT_ASSERT_EQUAL(patchSet.size(), 0);
        }
    }

    Y_UNIT_TEST(AddedSingle) {
        TString source(R"(
--- /dev/null 2018-11-08 14:36:09.883332277 +0300
+++ 123 2018-11-09 12:41:07.102448420 +0300
@@ -0,0 +1 @@
+2

)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 1);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "/dev/null");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "123");
        UNIT_ASSERT_EQUAL(diff.Removed.size(), 0);
        UNIT_ASSERT_EQUAL(diff.Added.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Added[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[0].Line, "2");
    }

    Y_UNIT_TEST(RemoveSingle) {
        TString source(R"(
--- 123 2018-11-09 12:41:07.102448420 +0300
+++ /dev/null 2018-11-08 14:36:09.883332277 +0300
@@ -1 +0,0 @@
-2

)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 1);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "123");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "/dev/null");
        UNIT_ASSERT_EQUAL(diff.Removed.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Added.size(), 0);
        UNIT_ASSERT_EQUAL(diff.Removed[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[0].Line, "2");
    }

    Y_UNIT_TEST(ReplaceSingle) {
        TString source(R"(
--- 123 2018-11-09 13:15:30.120930447 +0300
+++ 1234 2018-11-09 13:15:41.060990622 +0300
@@ -1 +1 @@
-test
+eggs

)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 1);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "123");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "1234");
        UNIT_ASSERT_EQUAL(diff.Removed.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Removed[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[0].Line, "test");
        UNIT_ASSERT_EQUAL(diff.Added.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Added[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[0].Line, "eggs");
    }

    Y_UNIT_TEST(ReplaceMultiline) {
        TString source(R"(
--- 123 2018-11-09 13:17:34.084946417 +0300
+++ 1234 2018-11-09 13:17:44.358336406 +0300
@@ -1,4 +1,4 @@
 test
-removed
-line
+added
+fline
 test
\ No newline at end of file
)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 1);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "123");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "1234");
        UNIT_ASSERT_EQUAL(diff.Removed.size(), 2);
        UNIT_ASSERT_EQUAL(diff.Removed[0].LineNo, 2);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[0].Line, "removed");
        UNIT_ASSERT_EQUAL(diff.Removed[1].LineNo, 3);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[1].Line, "line");
        UNIT_ASSERT_EQUAL(diff.Added.size(), 2);
        UNIT_ASSERT_EQUAL(diff.Added[0].LineNo, 2);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[0].Line, "added");
        UNIT_ASSERT_EQUAL(diff.Added[1].LineNo, 3);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[1].Line, "fline");
    }

    Y_UNIT_TEST(ReplaceMultiInSingleHunk) {
        TString source(R"(
--- 123 2018-11-09 13:37:45.064293724 +0300
+++ 1234 2018-11-09 13:38:09.617768350 +0300
@@ -1,5 +1,5 @@
 foo
-bar
+bar-replaced
 baz
 spam
-eggs
+eggs-replaced
)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 1);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "123");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "1234");
        UNIT_ASSERT_EQUAL(diff.Removed.size(), 2);
        UNIT_ASSERT_EQUAL(diff.Removed[0].LineNo, 2);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[0].Line, "bar");
        UNIT_ASSERT_EQUAL(diff.Removed[1].LineNo, 5);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[1].Line, "eggs");
        UNIT_ASSERT_EQUAL(diff.Added.size(), 2);
        UNIT_ASSERT_EQUAL(diff.Added[0].LineNo, 2);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[0].Line, "bar-replaced");
        UNIT_ASSERT_EQUAL(diff.Added[1].LineNo, 5);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[1].Line, "eggs-replaced");
    }

    Y_UNIT_TEST(ReplaceMultiHunks) {
        TString source(R"(
--- 123 2018-11-09 13:51:19.962333752 +0300
+++ 1234 2018-11-09 13:52:13.572644569 +0300
@@ -1,6 +1,6 @@
-0foo
+0foo-replaced
 0bar
-0baz
+0baz-replaced
 0spam
 0eggs
 1foo
@@ -9,7 +9,7 @@
 1spam
 1eggs
 2foo
-2bar
-2baz
+2bar-replaced
+2baz-replaced
 2spam
 2eggs
)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 1);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "123");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "1234");

        UNIT_ASSERT_EQUAL(diff.Removed.size(), 4);
        UNIT_ASSERT_EQUAL(diff.Removed[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[0].Line, "0foo");
        UNIT_ASSERT_EQUAL(diff.Removed[1].LineNo, 3);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[1].Line, "0baz");
        UNIT_ASSERT_EQUAL(diff.Removed[2].LineNo, 12);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[2].Line, "2bar");
        UNIT_ASSERT_EQUAL(diff.Removed[3].LineNo, 13);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[3].Line, "2baz");

        UNIT_ASSERT_EQUAL(diff.Added.size(), 4);
        UNIT_ASSERT_EQUAL(diff.Added[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[0].Line, "0foo-replaced");
        UNIT_ASSERT_EQUAL(diff.Added[1].LineNo, 3);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[1].Line, "0baz-replaced");
        UNIT_ASSERT_EQUAL(diff.Added[2].LineNo, 12);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[2].Line, "2bar-replaced");
        UNIT_ASSERT_EQUAL(diff.Added[3].LineNo, 13);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[3].Line, "2baz-replaced");
    }

    Y_UNIT_TEST(ReplaceMultiFiles) {
        TString source(R"(
diff --git a/1 b/1
index d00491f..b4de394 100644
--- a/1
+++ b/1
@@ -1 +1 @@
-1
+11
diff --git a/2 b/2
index 0cfbf08..2bd5a0a 100644
--- a/2
+++ b/2
@@ -1 +1 @@
-2
+22
)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 2);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "a/1");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "b/1");

        UNIT_ASSERT_EQUAL(diff.Removed.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Removed[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[0].Line, "1");

        UNIT_ASSERT_EQUAL(diff.Added.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Added[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[0].Line, "11");

        diff = patchSet[1];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "a/2");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "b/2");

        UNIT_ASSERT_EQUAL(diff.Removed.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Removed[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Removed[0].Line, "2");

        UNIT_ASSERT_EQUAL(diff.Added.size(), 1);
        UNIT_ASSERT_EQUAL(diff.Added[0].LineNo, 1);
        UNIT_ASSERT_STRINGS_EQUAL(diff.Added[0].Line, "22");
    }

    Y_UNIT_TEST(FilenamesOnly) {
        TString source(R"(
--- /a
+++ /b
@@ -1 +1 @@
-test
+eggs

)");
        auto patchSet = ParsePatchSet(source);
        UNIT_ASSERT_EQUAL(patchSet.size(), 1);
        auto diff = patchSet[0];

        UNIT_ASSERT_STRINGS_EQUAL(diff.SourceFile, "/a");
        UNIT_ASSERT_STRINGS_EQUAL(diff.TargetFile, "/b");
    }
}
