#include <security/ant-secret/snooper/cpp/snooper.h>
#include <security/ant-secret/snooper/cpp/searcher.h>

#include <library/cpp/getopt/last_getopt.h>

#include <util/system/datetime.h>
#include <util/datetime/base.h>
#include <util/random/fast.h>
#include <util/string/builder.h>
#include <util/system/hp_timer.h>

namespace {
    TFastRng64 rng(1337);

    TString RandomString(size_t size, const TString& innerString = TString()) {
        TStringBuilder result;
        result.reserve(size);

        size_t total = (size - innerString.length()) / 2;

        for (size_t i = 0; i < total; ++i) {
            result << char(rng.GenRand());
        }

        result << innerString;

        for (size_t i = 0; i < total; ++i) {
            result << char(rng.GenRand());
        }

        return result;
    }

    size_t search(NSnooper::TSnooper& snooper, const TDeque<TString>& lines) {
        size_t result = 0;
        auto searcher = snooper.Searcher();
        for (auto&& line : lines) {
            result += searcher->Search(line).size();
        }

        return result;
    }

}

int main(int argc, char** argv) {
    auto opts = NLastGetopt::TOpts::Default();
    opts.SetTitle("Snooper perf test");

    NLastGetopt::TOptsParseResult args(&opts, argc, argv);
    NSnooper::TSnooper snooper;

    {
        size_t count = 1e3;
        size_t len = 1e3;

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len, "test"));
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/o secrets]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e6;
        size_t len = 1e3;

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len, "test"));
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/o secrets]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e3;
        size_t len = 1e6;

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len, "test"));
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/o secrets]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e3;
        size_t len = 1e3;

        TString cookies = "yandexuid=1234; "
                          "Session_id=XXXX; "
                          "sessionid2=XXXX; "
                          "something=brown;";

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len, cookies));
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/ inner masked-secrets]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e3;
        size_t len = 1e6;

        TString cookies = "yandexuid=1234; "
                          "Session_id=XXXX; "
                          "sessionid2=XXXX; "
                          "something=brown;";

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len, cookies));
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/ inner masked-secrets]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e3;
        size_t len = 1e3;

        TString cookies = "yandexuid=1234; "
                          "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM; "
                          "sessionid2=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY; "
                          "something=brown;";

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len, cookies));
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/ inner secrets]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e3;
        size_t len = 1e6;

        TString cookies = "yandexuid=1234; "
                          "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM; "
                          "sessionid2=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY; "
                          "something=brown;";

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len, cookies));
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/ inner secrets]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e3;
        size_t len = 1e3;

        TString cookies = "yandexuid=1234; "
                          "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM; "
                          "sessionid2=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY; "
                          "something=brown;";

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len - cookies.length()) + cookies);
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/ suffix secret]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }

    {
        size_t count = 1e3;
        size_t len = 1e3;

        TString cookies = "yandexuid=1234; "
                          "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM; "
                          "sessionid2=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY; "
                          "something=brown;";

        TDeque<TString> lines;
        for (size_t i = 0; i < count; ++i) {
            lines.push_back(RandomString(len - cookies.length()) + cookies);
        }

        THPTimer timer;
        size_t founds = search(snooper, lines);
        Cout
            << "[" << count << " * " << len << ", w/ prefix secret]: "
            << founds << " secrets in " << timer.Passed()
            << Endl;
    }
}