#include <security/ant-secret/snooper/cpp/snooper.h>
#include <security/ant-secret/snooper/cpp/secret.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/threading/future/async.h>

#include <util/generic/string.h>
#include <util/thread/pool.h>

Y_UNIT_TEST_SUITE(TSnooperTests) {
    using namespace NSnooper;

    namespace {
        struct TSimpleCase {
            TString Content;
            TString MaskedContent;
            TSecretList ExpectedSecrets;
        };

        void SecretsEqual(const TSecretList& a, const TSecretList& b, const TString& testName) {
            UNIT_ASSERT_VALUES_EQUAL_C(a.size(), b.size(), testName);
            for (decltype(a.size()) i = 0; i < a.size(); ++i) {
                bool found = false;
                TSecret secret;
                for (decltype(b.size()) k = 0; k < b.size(); ++k) {
                    if (a[i].Secret == b[k].Secret) {
                        found = true;
                        secret = b[k];
                    }
                }

                UNIT_ASSERT_C(
                    found,
                    "secret " << a[i].Secret << " was not found for test " << testName);
                UNIT_ASSERT_VALUES_EQUAL_C(
                    a[i].Type,
                    secret.Type,
                    "secret " << a[i].Secret << " type mismatch for test " << testName);
                UNIT_ASSERT_VALUES_EQUAL_C(
                    a[i].MaskPos.From,
                    secret.MaskPos.From,
                    "secret " << a[i].Secret << " mask pos for test " << testName);
                UNIT_ASSERT_VALUES_EQUAL_C(
                    a[i].MaskPos.Len,
                    secret.MaskPos.Len,
                    "secret " << a[i].Secret << " mask len for test " << testName);
                UNIT_ASSERT_VALUES_EQUAL_C(
                    a[i].SecretPos.From,
                    secret.SecretPos.From,
                    "secret " << a[i].Secret << " secret pos for test " << testName);
                UNIT_ASSERT_VALUES_EQUAL_C(
                    a[i].SecretPos.Len,
                    secret.SecretPos.Len,
                    "secret " << a[i].Secret << " secret len for test " << testName);
            }
        }
    }

    Y_UNIT_TEST(TestOAuth) {
        THashMap<TString, TSimpleCase> cases = {
            {
                "embedded_v2",
                TSimpleCase{
                    .Content = R"(Authorization: OAuth AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3)",
                    .MaskedContent = R"(Authorization: OAuth AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX)",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3",
                            .SecretPos = {
                                .From =   21,
                                .Len = 39,
                            },
                            .MaskPos = {
                                .From = 39,
                                .Len = 21,
                            }},
                    }},
            },
            {
                "embedded_v2_encoded",
                TSimpleCase{
                    .Content = R"(%7B%22token%22%3A%22Authorization%3A%20OAuth%20AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3%22%7D,-ua=Mozilla%2F5.0%20(Macintosh%3B%20Intel%20Mac%20OS%20X%2010_14_6))",
                    .MaskedContent = R"(%7B%22token%22%3A%22Authorization%3A%20OAuth%20AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX%22%7D,-ua=Mozilla%2F5.0%20(Macintosh%3B%20Intel%20Mac%20OS%20X%2010_14_6))",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3",
                            .SecretPos = {
                                .From = 47,
                                .Len = 39,
                            },
                            .MaskPos = {
                                .From = 65,
                                .Len = 21,
                            }},
                    }},
            },
            {
                "stateles",
                TSimpleCase{
                    .Content = R"(token=1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q)",
                    .MaskedContent = R"(token=1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.XXXXXXXXXXXXXXXXXXXXXX)",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q",
                            .SecretPos = {
                                .From =   6,
                                .Len = 87,
                            },
                            .MaskPos = {
                                .From = 71,
                                .Len = 22,
                            }},
                    }},
            },
            {
                "stateles_encoded",
                TSimpleCase{
                    .Content = "token=1%2e0%2e123%2e2222222222%2e1111222233000%2e12350%2eYRf1SNShCFuRY56E%2eKO0b4NfG%2ek0sB3ImkZb5MarhDjgyU5Q&userip=1.1.1.1",
                    .MaskedContent = "token=1%2e0%2e123%2e2222222222%2e1111222233000%2e12350%2eYRf1SNShCFuRY56E%2eKO0b4NfG%2eXXXXXXXXXXXXXXXXXXXXXX&userip=1.1.1.1",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q",
                            .SecretPos = {
                                .From =   6,
                                .Len = 103,
                            },
                            .MaskPos = {
                                .From = 87,
                                .Len = 22,
                            }},
                    }},
            },
            {
                "embedded_v3",
                TSimpleCase{
                    .Content = R"(token=y0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSA)",
                    .MaskedContent = R"(token=y0_AQAAAAAAAYiUAAAAgQAAAAA63mixXXXXXXXXXXXXXXXXXXXXXXXXXXX)",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "y0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSA",
                            .SecretPos = {
                                .From = 6,
                                .Len = 58,
                            },
                            .MaskPos = {
                                .From = 37,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "text",
                TSimpleCase{
                    .Content = R"(AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3 1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q)",
                    .MaskedContent = R"(AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX 1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.XXXXXXXXXXXXXXXXXXXXXX)",
                    .ExpectedSecrets = {
                        {
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3",
                            .SecretPos = {
                                .From = 0,
                                .Len = 39,
                            },
                            .MaskPos = {
                                .From = 18,
                                .Len = 21,
                            }
                        },
                        {
                            .Type = ESecretType::YOAuth,
                            .Secret = "1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q",
                            .SecretPos = {
                                .From = 40,
                                .Len = 87,
                            },
                            .MaskPos = {
                                .From = 105,
                                .Len = 22,
                            }
                        },
                    },
                },
            },
            {
                "token_only",
                TSimpleCase{
                    .Content = R"(AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3)",
                    .MaskedContent = R"(AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX)",
                    .ExpectedSecrets = {
                        {
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3",
                            .SecretPos = {
                                .From = 0,
                                .Len = 39,
                            },
                            .MaskPos = {
                                .From = 18,
                                .Len = 21,
                            }
                        },
                    },
                },
            },
            {
                "embedded_v2_escaped",
                TSimpleCase{
                    .Content = R"(token\=AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3&userip\=1.1.1.1)",
                    .MaskedContent = R"(token\=AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX&userip\=1.1.1.1)",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3",
                            .SecretPos = {
                                .From =   7,
                                .Len = 39,
                            },
                            .MaskPos = {
                                .From = 25,
                                .Len = 21,
                            }},
                    }},
            },
            {
                "embedded_v2_separators",
                TSimpleCase{
                    .Content = "token=AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm0\n"
                               "TOKEN=AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm1\n"
                               "'SOME_TOKEN\\\\\\'   => \\\\\\'AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm2\\\\\\'"
                               "\"SOME_TOKEN\\\\\" => \\\\\"AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3\\\\\"",
                    .MaskedContent = "token=AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX\n"
                                     "TOKEN=AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX\n"
                                     "'SOME_TOKEN\\\\\\'   => \\\\\\'AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX\\\\\\'"
                                     "\"SOME_TOKEN\\\\\" => \\\\\"AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX\\\\\"",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm0",
                            .SecretPos = {
                                    .From =   6,
                                    .Len = 39,
                            },
                            .MaskPos = {
                                    .From = 24,
                                    .Len = 21,
                            }},
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm1",
                            .SecretPos = {
                                    .From =   52,
                                    .Len = 39,
                            },
                            .MaskPos = {
                                    .From = 70,
                                    .Len = 21,
                            }},
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm2",
                            .SecretPos = {
                                    .From =   117,
                                    .Len = 39,
                            },
                            .MaskPos = {
                                    .From = 135,
                                    .Len = 21,
                            }},
                        TSecret{
                            .Type = ESecretType::YOAuth,
                            .Secret = "AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3",
                            .SecretPos = {
                                    .From =   181,
                                    .Len = 39,
                            },
                            .MaskPos = {
                                    .From = 199,
                                    .Len = 21,
                            }},
                    }},
            },
        };

        TSnooper snooper;
        auto searcher = snooper.Searcher(NSecret::ESecretType::YOAuth);
        for (auto&& [name, testCase] : cases) {
            auto results = searcher->Search(testCase.Content);
            SecretsEqual(testCase.ExpectedSecrets, results, name);
            if (testCase.MaskedContent) {
                TString originalContent = testCase.Content.copy();
                TString result = testCase.Content;
                auto masked = searcher->Mask(result);
                UNIT_ASSERT_C(masked, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(testCase.MaskedContent, result, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(originalContent, testCase.Content, "test " << name);
            }
        }
    }

    Y_UNIT_TEST(TestSessionID) {
        THashMap<TString, TSimpleCase> cases = {
            {
                "sess_only_external",
                TSimpleCase{
                    .Content = "3:1653555178.5.0.1649862799022:PWqlXw:e.1.2:1|253917951.0.302|3:252899.751979.Tty8UAAAAAAAAGNJbaO_YGk_ujg",
                    .MaskedContent = "3:1653555178.5.0.1649862799022:PWqlXw:e.1.2:1|253917951.0.302|3:252899.751979.XXXXXXXXXXXXXXXXXXXXXXXXXXX",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1653555178.5.0.1649862799022:PWqlXw:e.1.2:1|253917951.0.302|3:252899.751979.Tty8UAAAAAAAAGNJbaO_YGk_ujg",
                            .SecretPos = {
                                .From =   0,
                                .Len = 105,
                            },
                            .MaskPos = {
                                .From = 78,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "sess_external_with_extended_kv",
                TSimpleCase{
                    .Content = "3:1653657845.5.0.1654540372341:NbURsg:39.1.2:3.100:NbURsg.101:1634557845.102:1653657845|1120000000249440.117473.4002.2:117473|5:191027.624461.-ki8okoGQVmAAAAAAAAiTKZx7oc",
                    .MaskedContent = "3:1653657845.5.0.1654540372341:NbURsg:39.1.2:3.100:NbURsg.101:1634557845.102:1653657845|1120000000249440.117473.4002.2:117473|5:191027.624461.XXXXXXXXXXXXXXXXXXXXXXXXXXX",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1653657845.5.0.1654540372341:NbURsg:39.1.2:3.100:NbURsg.101:1634557845.102:1653657845|1120000000249440.117473.4002.2:117473|5:191027.624461.-ki8okoGQVmAAAAAAAAiTKZx7oc",
                            .SecretPos = {
                                .From =   0,
                                .Len = 169,
                            },
                            .MaskPos = {
                                .From = 142,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "sess_only_internal",
                TSimpleCase{
                    .Content = "3:1653572691.5.0.1649864264115:BAABAAAAAAATEoGwuAYCKg:39.1.2:3|1120000000038691.3708427.2002.2:3708427|5:190979.992482.9dD2Qvfv5LoAAAAAAAAAAAAAAjI",
                    .MaskedContent = "3:1653572691.5.0.1649864264115:BAABAAAAAAATEoGwuAYCKg:39.1.2:3|1120000000038691.3708427.2002.2:3708427|5:190979.992482.XXXXXXXXXXXXXXXXXXXXXXXXXXX",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1653572691.5.0.1649864264115:BAABAAAAAAATEoGwuAYCKg:39.1.2:3|1120000000038691.3708427.2002.2:3708427|5:190979.992482.9dD2Qvfv5LoAAAAAAAAAAAAAAjI",
                            .SecretPos = {
                                .From =   0,
                                .Len = 146,
                            },
                            .MaskPos = {
                                .From = 119,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "sessid2_only_external",
                TSimpleCase{
                    .Content = "3:1653555178.5.0.1649862799022:PWqlXw:e.1.2:1.499:1|253917951.0.302|3:252899.416214.iaPip-CSUjSCAAAAAAAAAAAA8XM",
                    .MaskedContent = "3:1653555178.5.0.1649862799022:PWqlXw:e.1.2:1.499:1|253917951.0.302|3:252899.416214.XXXXXXXXXXXXXXXXXXXXXXXXXXX",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1653555178.5.0.1649862799022:PWqlXw:e.1.2:1.499:1|253917951.0.302|3:252899.416214.iaPip-CSUjSCAAAAAAAAAAAA8XM",
                            .SecretPos = {
                                .From =   0,
                                .Len = 111,
                            },
                            .MaskPos = {
                                .From = 84,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "sessid_multi_internal",
                TSimpleCase{
                    .Content = "3:1653581899.5.1.1649864264115:BAABAAAAAAATEoGwuAYCKg:39.1.2:3|1120000000038691.3708427.2002.2:3708427|1120000000407792.3717635.2002.2:3717635|5:190985.403398.g_RkVNXD9EAAAAAAAAAAAFSfol4",
                    .MaskedContent = "3:1653581899.5.1.1649864264115:BAABAAAAAAATEoGwuAYCKg:39.1.2:3|1120000000038691.3708427.2002.2:3708427|1120000000407792.3717635.2002.2:3717635|5:190985.403398.XXXXXXXXXXXXXXXXXXXXXXXXXXX",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1653581899.5.1.1649864264115:BAABAAAAAAATEoGwuAYCKg:39.1.2:3|1120000000038691.3708427.2002.2:3708427|1120000000407792.3717635.2002.2:3717635|5:190985.403398.g_RkVNXD9EAAAAAAAAAAAFSfol4",
                            .SecretPos = {
                                .From =   0,
                                .Len = 186,
                            },
                            .MaskPos = {
                                .From = 159,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "cookies",
                TSimpleCase{
                    .Content = "yandexuid=1234; "
                               "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:ff.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM; "
                               "session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:ff.1|253917951.0.302|126480966.1391717.2.2:1391717|5:204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM; "
                               "sessionid2=3:1566815113.5.0.1563223444253:-DwABAAAAAAAHZICwuAYCKg:FF.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY; "
                               "something=brown",
                    .MaskedContent = "yandexuid=1234; "
                                     "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:ff.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:ff.1|253917951.0.302|126480966.1391717.2.2:1391717|5:204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "sessionid2=3:1566815113.5.0.1563223444253:-DwABAAAAAAAHZICwuAYCKg:FF.1|253917951.0.302|126480966.-1.0|204187.563454.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "something=brown",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:ff.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                            .SecretPos = {
                                .From =   27,
                                .Len = 146,
                            },
                            .MaskPos = {
                                .From = 146,
                                .Len = 27,
                            }},
                        TSecret{
                                .Type = ESecretType::YSession,
                                .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:ff.1|253917951.0.302|126480966.1391717.2.2:1391717|5:204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                                .SecretPos = {
                                        .From =   186,
                                        .Len = 148,
                                },
                                .MaskPos = {
                                        .From = 307,
                                        .Len = 27,
                                }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:-DwABAAAAAAAHZICwuAYCKg:FF.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY",
                            .SecretPos = {
                                .From =   347,
                                .Len = 132,
                            },
                            .MaskPos = {
                                .From = 452,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "json_encoded",
                TSimpleCase{
                    .Content = "additional=%7B%22session_id%22%3A%223%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.1391717.2.2%3A1391717%7C204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM%22%7D,-ua=Mozilla%2F5.0%20(Macintosh%3B%20Intel%20Mac%20OS%20X",
                    .MaskedContent = "additional=%7B%22session_id%22%3A%223%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.1391717.2.2%3A1391717%7C204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX%22%7D,-ua=Mozilla%2F5.0%20(Macintosh%3B%20Intel%20Mac%20OS%20X",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                            .SecretPos = {
                                .From = 36,
                                .Len = 160,
                            },
                            .MaskPos = {
                                .From = 169,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "cookies_multiaccount",
                TSimpleCase{
                    .Content = "yandexuid=654321; "
                               "Session_id=3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.ySsdPJeQSM1Zoj32qOMvTKSWVxM; "
                               "sessionid2=3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.OJbvg_ahfMoFX22xPvf-iEoVNsk; "
                               "something=brown",
                    .MaskedContent = "yandexuid=654321; "
                                     "Session_id=3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "sessionid2=3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "something=brown",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.ySsdPJeQSM1Zoj32qOMvTKSWVxM",
                            .SecretPos = {
                                .From =   29,
                                .Len = 146,
                            },
                            .MaskPos = {
                                .From = 148,
                                .Len = 27,
                            }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.OJbvg_ahfMoFX22xPvf-iEoVNsk",
                            .SecretPos = {
                                .From =   188,
                                .Len = 130,
                            },
                            .MaskPos = {
                                .From = 291,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "bb_request",
                TSimpleCase{
                    .Content = "http://blackbox.yandex.net/blackbox?"
                               "method=sessionid"
                               "&host=yandex.ru"
                               "&sessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.1391717.2.2%3A1391717%7C204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM"
                               "&sslsessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.-1.0%7C204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY"
                               "&userip=12.12.12.12",
                    .MaskedContent = "http://blackbox.yandex.net/blackbox?"
                                     "method=sessionid"
                                     "&host=yandex.ru"
                                     "&sessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.1391717.2.2%3A1391717%7C204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX"
                                     "&sslsessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.-1.0%7C204187.563454.XXXXXXXXXXXXXXXXXXXXXXXXXXX"
                                     "&userip=12.12.12.12",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                            .SecretPos = {
                                .From =   78,
                                .Len = 160,
                            },
                            .MaskPos = {
                                .From = 211,
                                .Len = 27,
                            }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY",
                            .SecretPos = {
                                .From =   252,
                                .Len = 143,
                            },
                            .MaskPos = {
                                .From = 368,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "bb_request_json",
                TSimpleCase{
                    .Content = R"({"method":"sessionid","host":"yandex.ru","sessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM","sslsessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY","userip":"12.12.12.12"})",
                    .MaskedContent = R"({"method":"sessionid","host":"yandex.ru","sessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX","sslsessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.XXXXXXXXXXXXXXXXXXXXXXXXXXX","userip":"12.12.12.12"})",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                            .SecretPos = {
                                .From =   54,
                                .Len = 146,
                            },
                            .MaskPos = {
                                .From = 173,
                                .Len = 27,
                            }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY",
                            .SecretPos = {
                                .From =   218,
                                .Len = 131,
                            },
                            .MaskPos = {
                                .From = 322,
                                .Len = 27,
                            }},
                    }},
            },
        };

        TSnooper snooper;
        auto searcher = snooper.Searcher(NSecret::ESecretType::YSession);
        for (auto&& [name, testCase] : cases) {
            auto results = searcher->Search(testCase.Content);
            SecretsEqual(testCase.ExpectedSecrets, results, name);
            if (testCase.MaskedContent) {
                TString originalContent = testCase.Content.copy();
                TString result = testCase.Content;
                auto masked = searcher->Mask(result);
                UNIT_ASSERT_C(masked, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(testCase.MaskedContent, result, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(originalContent, testCase.Content, "test " << name);
            }
        }
    }

    Y_UNIT_TEST(TestSessionIDThreaded) {
        THashMap<TString, TSimpleCase> cases =
        {
            {
                "cookies",
                TSimpleCase{
                    .Content = "yandexuid=1234; "
                               "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM; "
                               "sessionid2=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY; "
                               "something=brown",
                    .MaskedContent = "yandexuid=1234; "
                                     "Session_id=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "sessionid2=3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "something=brown",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                            .SecretPos = {
                                .From =   27,
                                .Len = 146,
                            },
                            .MaskPos = {
                                .From = 146,
                                .Len = 27,
                            }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY",
                            .SecretPos = {
                                .From =   186,
                                .Len = 131,
                            },
                            .MaskPos = {
                                .From = 290,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "cookies_multiaccount",
                TSimpleCase{
                    .Content = "yandexuid=654321; "
                               "Session_id=3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.ySsdPJeQSM1Zoj32qOMvTKSWVxM; "
                               "sessionid2=3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.OJbvg_ahfMoFX22xPvf-iEoVNsk; "
                               "something=brown",
                    .MaskedContent = "yandexuid=654321; "
                                     "Session_id=3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "sessionid2=3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.XXXXXXXXXXXXXXXXXXXXXXXXXXX; "
                                     "something=brown",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.ySsdPJeQSM1Zoj32qOMvTKSWVxM",
                            .SecretPos = {
                                .From =   29,
                                .Len = 146,
                            },
                            .MaskPos = {
                                .From = 148,
                                .Len = 27,
                            }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.OJbvg_ahfMoFX22xPvf-iEoVNsk",
                            .SecretPos = {
                                .From =   188,
                                .Len = 130,
                            },
                            .MaskPos = {
                                .From = 291,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "bb_request",
                TSimpleCase{
                    .Content = "http://blackbox.yandex.net/blackbox?"
                               "method=sessionid"
                               "&host=yandex.ru"
                               "&sessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.1391717.2.2%3A1391717%7C204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM"
                               "&sslsessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.-1.0%7C204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY"
                               "&userip=12.12.12.12",
                    .MaskedContent = "http://blackbox.yandex.net/blackbox?"
                                     "method=sessionid"
                                     "&host=yandex.ru"
                                     "&sessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.1391717.2.2%3A1391717%7C204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX"
                                     "&sslsessionid=3%3A1566815113.5.0.1563223444253%3ADwABAAAAAAAHZICwuAYCKg%3A14.1%7C253917951.0.302%7C126480966.-1.0%7C204187.563454.XXXXXXXXXXXXXXXXXXXXXXXXXXX"
                                     "&userip=12.12.12.12",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                            .SecretPos = {
                                .From =   78,
                                .Len = 160,
                            },
                            .MaskPos = {
                                .From = 211,
                                .Len = 27,
                            }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY",
                            .SecretPos = {
                                .From =   252,
                                .Len = 143,
                            },
                            .MaskPos = {
                                .From = 368,
                                .Len = 27,
                            }},
                    }},
            },
            {
                "bb_request_json",
                TSimpleCase{
                    .Content = R"({"method":"sessionid","host":"yandex.ru","sessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM","sslsessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY","userip":"12.12.12.12"})",
                    .MaskedContent = R"({"method":"sessionid","host":"yandex.ru","sessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.XXXXXXXXXXXXXXXXXXXXXXXXXXX","sslsessionid":"3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.XXXXXXXXXXXXXXXXXXXXXXXXXXX","userip":"12.12.12.12"})",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.1391717.2.2:1391717|204187.211232.T2yhhadLG25l1IrKGp86_rEyIdM",
                            .SecretPos = {
                                .From =   54,
                                .Len = 146,
                            },
                            .MaskPos = {
                                .From = 173,
                                .Len = 27,
                            }},
                        TSecret{
                            .Type = ESecretType::YSession,
                            .Secret = "3:1566815113.5.0.1563223444253:DwABAAAAAAAHZICwuAYCKg:14.1|253917951.0.302|126480966.-1.0|204187.563454.Zo7YHD-LF1JRjmzVK96xGRzwAQY",
                            .SecretPos = {
                                .From =   218,
                                .Len = 131,
                            },
                            .MaskPos = {
                                .From = 322,
                                .Len = 27,
                            }},
                    }},
            },
        };

        TSnooper snooper;
        TThreadPool queue{TThreadPool::TParams().SetBlocking(false).SetCatching(false)};
        queue.Start(cases.size());
        TVector<NThreading::TFuture<void>> futures;
        for (auto&& current : cases) {
            TString name = current.first;
            TSimpleCase testCase = current.second;

            futures.push_back(
                NThreading::Async([&snooper, name, testCase] {
                    auto searcher = snooper.Searcher(NSecret::ESecretType::YSession);
                    auto results = searcher->Search(testCase.Content);
                    SecretsEqual(testCase.ExpectedSecrets, results, name);
                    if (testCase.MaskedContent) {
                        TString originalContent = testCase.Content.copy();
                        TString result = testCase.Content;
                        auto masked = searcher->Mask(result);
                        UNIT_ASSERT_C(masked, "test " << name);
                        UNIT_ASSERT_VALUES_EQUAL_C(testCase.MaskedContent, result, "test " << name);
                        UNIT_ASSERT_VALUES_EQUAL_C(originalContent, testCase.Content, "test " << name);
                    }
                }, queue)
            );
        }

        NThreading::WaitExceptionOrAll(futures).GetValueSync();
    }

    Y_UNIT_TEST(TestTVM) {
        THashMap<TString, TSimpleCase> cases = {
            {
                "service_ticket",
                TSimpleCase{
                    .Content = "X-Ya-Service-Ticket: 3:serv:CBAQ__________9_IgQIARAD:WOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rAWOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rA",
                    .MaskedContent = "X-Ya-Service-Ticket: 3:serv:CBAQ__________9_IgQIARAD:XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::TVMTicket,
                            .Secret = "3:serv:CBAQ__________9_IgQIARAD:WOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rAWOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rA",
                            .SecretPos = {
                                .From = 21,
                                .Len = 374,
                            },
                            .MaskPos = {
                                .From = 53,
                                .Len = 342,
                            }},
                    }},
            },
            {
                "service_ticket_enc",
                TSimpleCase{
                        .Content = "3%3Aserv%3ACBAQ__________9_IgQIARAD%3AWOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rAWOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rA",
                        .MaskedContent = "3%3Aserv%3ACBAQ__________9_IgQIARAD%3AXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
                        .ExpectedSecrets = {
                                TSecret{
                                        .Type = ESecretType::TVMTicket,
                                        .Secret = "3:serv:CBAQ__________9_IgQIARAD:WOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rAWOb6ij6ECsFfL2_-mYN_IhDhGQG2h3_PmpVNGw-Y1iPByS9nn88s8RAcVXftA0TkAAKeG922aAhBq_73aIHXGI7Fvr-az-B2hSaH5IRbfszuE_EeR6ypEzRBEkWNrxJ2t7ysOlrc0p_ozN4iKwH-XJGPt0kIAgbb4hQjYTqY5rA",
                                        .SecretPos = {
                                                .From = 0,
                                                .Len = 380,
                                        },
                                        .MaskPos = {
                                                .From = 38,
                                                .Len = 342,
                                        }},
                        }},
            },
            {
                "service_ticket_python_hell",
                TSimpleCase{
                        .Content = R"({"X-Ya-Service-Ticket": "u'3:serv:CLupARCJ8Z-UBiIICO6behDum3o:Pm0XkkLNAqivijEK5DQMc0R6pGBQX3sTYbI6KaeJfC34iWjrGqmhwf2x-dlljluMVY-A6CaQwfeexxSZWDz2Qc1HHQVZMmFtgU61BSVFvZu3_hsvcN4Ee3hE62WB20RC245E9AGZcms_zl093bVerSuDVFI0MmpwhDh9UpBYxmmjgCc-4SO3zBvsdyTUC_Y2vmbPDwNjUC5OrBIqsPa6yi4jJH6Uh0rbcNiLpLfC1oyZ2bkux_8D4wj1Izxv4kAKisjisApXmxlDLvcUAAAAAASMmH71OZ3ab8RTXQm6IF0MSXuXYOSe8Gg6Xy8NEVW0L1HolqShFVbPSkDSiK_ppg'"})",
                        .MaskedContent = R"({"X-Ya-Service-Ticket": "u'3:serv:CLupARCJ8Z-UBiIICO6behDum3o:XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX'"})",
                        .ExpectedSecrets = {
                                TSecret{
                                        .Type = ESecretType::TVMTicket,
                                        .Secret = "3:serv:CLupARCJ8Z-UBiIICO6behDum3o:Pm0XkkLNAqivijEK5DQMc0R6pGBQX3sTYbI6KaeJfC34iWjrGqmhwf2x-dlljluMVY-A6CaQwfeexxSZWDz2Qc1HHQVZMmFtgU61BSVFvZu3_hsvcN4Ee3hE62WB20RC245E9AGZcms_zl093bVerSuDVFI0MmpwhDh9UpBYxmmjgCc-4SO3zBvsdyTUC_Y2vmbPDwNjUC5OrBIqsPa6yi4jJH6Uh0rbcNiLpLfC1oyZ2bkux_8D4wj1Izxv4kAKisjisApXmxlDLvcUAAAAAASMmH71OZ3ab8RTXQm6IF0MSXuXYOSe8Gg6Xy8NEVW0L1HolqShFVbPSkDSiK_ppg",
                                        .SecretPos = {
                                                .From = 27,
                                                .Len = 377,
                                        },
                                        .MaskPos = {
                                                .From = 62,
                                                .Len = 342,
                                        }},
                        }},
            },
            {
                "user_ticket",
                TSimpleCase{
                    .Content = "X-Ya-User-Ticket: 3:user:CLWpARDCop2UBhowCgYIpuC1lw8QpuC1lw8aDGJiOnNlc3NndWFyZBoMYmI6c2Vzc2lvbmlkIJTceigB:WKobNyDjtC1MYCb8nSkiWUbjeAftbdOpKeb4JGasYo_UBjdNtn__WmKke3Ii_su9iILYvM4ol2TLi4KELcKiLG4tBkW8k4qXcHY-Xn9mekHKd1m89sgnW04iUewNw5dGaM31tXFPe4t7ZGcxV803uTaBnBdQ12eBbP2ESI8t8wMCzD0Oejc7roNwnK_oaQFttZmPSxBWSbxibOle_qF-kQISyIu1-q7B00hhfwBp7A791qJpn8nfeFNWvDtOWsUVXn5hp0_RUo5aQCxSO0b-AiUR8Olg4CxF37aBnBVJsdfmoMZ7PgMTpGec-cvHKKni1Sd1h1cCj90KbHE7uPEocg",
                    .MaskedContent = "X-Ya-User-Ticket: 3:user:CLWpARDCop2UBhowCgYIpuC1lw8QpuC1lw8aDGJiOnNlc3NndWFyZBoMYmI6c2Vzc2lvbmlkIJTceigB:XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::TVMTicket,
                            .Secret = "3:user:CLWpARDCop2UBhowCgYIpuC1lw8QpuC1lw8aDGJiOnNlc3NndWFyZBoMYmI6c2Vzc2lvbmlkIJTceigB:WKobNyDjtC1MYCb8nSkiWUbjeAftbdOpKeb4JGasYo_UBjdNtn__WmKke3Ii_su9iILYvM4ol2TLi4KELcKiLG4tBkW8k4qXcHY-Xn9mekHKd1m89sgnW04iUewNw5dGaM31tXFPe4t7ZGcxV803uTaBnBdQ12eBbP2ESI8t8wMCzD0Oejc7roNwnK_oaQFttZmPSxBWSbxibOle_qF-kQISyIu1-q7B00hhfwBp7A791qJpn8nfeFNWvDtOWsUVXn5hp0_RUo5aQCxSO0b-AiUR8Olg4CxF37aBnBVJsdfmoMZ7PgMTpGec-cvHKKni1Sd1h1cCj90KbHE7uPEocg",
                            .SecretPos = {
                                .From =   18,
                                .Len = 430,
                            },
                            .MaskPos = {
                                .From = 106,
                                .Len = 342,
                            }},
                    }},
            },
            {
                "user_ticket_python_hell",
                TSimpleCase{
                        .Content = R"lit({"user_ticket": "'3:user:CLWpARDCop2UBhowCgYIpuC1lw8QpuC1lw8aDGJiOnNlc3NndWFyZBoMYmI6c2Vzc2lvbmlkIJTceigB:WKobNyDjtC1MYCb8nSkiWUbjeAftbdOpKeb4JGasYo_UBjdNtn__WmKke3Ii_su9iILYvM4ol2TLi4KELcKiLG4tBkW8k4qXcHY-Xn9mekHKd1m89sgnW04iUewNw5dGaM31tXFPe4t7ZGcxV803uTaBnBdQ12eBbP2ESI8t8wMCzD0Oejc7roNwnK_oaQFttZmPSxBWSbxibOle_qF-kQISyIu1-q7B00hhfwBp7A791qJpn8nfeFNWvDtOWsUVXn5hp0_RUo5aQCxSO0b-AiUR8Olg4CxF37aBnBVJsdfmoMZ7PgMTpGec-cvHKKni1Sd1h1cCj90KbHE7uPEocg'", "e": "TicketParsingException('Invalid BlackBox environment', <Status.InvalidBlackboxEnv: 2>, 'ticket_type=user;expiration_time=1653035330;scope=bb:sessguard;scope=bb:sessionid;default_uid=4075647014;uid=4075647014;env=Test;')"}}, {"loc": "/place/db/iss3/instances/jfmtftpdiclxzmoz_ydo_backend_request_extractor_yp_test_IuFlpMmggxE/ydo/libs/tvm/tvm.py:77 check_user_ticket_and_get_default_uid()"})lit",
                        .MaskedContent = R"lit({"user_ticket": "'3:user:CLWpARDCop2UBhowCgYIpuC1lw8QpuC1lw8aDGJiOnNlc3NndWFyZBoMYmI6c2Vzc2lvbmlkIJTceigB:XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX'", "e": "TicketParsingException('Invalid BlackBox environment', <Status.InvalidBlackboxEnv: 2>, 'ticket_type=user;expiration_time=1653035330;scope=bb:sessguard;scope=bb:sessionid;default_uid=4075647014;uid=4075647014;env=Test;')"}}, {"loc": "/place/db/iss3/instances/jfmtftpdiclxzmoz_ydo_backend_request_extractor_yp_test_IuFlpMmggxE/ydo/libs/tvm/tvm.py:77 check_user_ticket_and_get_default_uid()"})lit",
                        .ExpectedSecrets = {
                                TSecret{
                                        .Type = ESecretType::TVMTicket,
                                        .Secret = "3:user:CLWpARDCop2UBhowCgYIpuC1lw8QpuC1lw8aDGJiOnNlc3NndWFyZBoMYmI6c2Vzc2lvbmlkIJTceigB:WKobNyDjtC1MYCb8nSkiWUbjeAftbdOpKeb4JGasYo_UBjdNtn__WmKke3Ii_su9iILYvM4ol2TLi4KELcKiLG4tBkW8k4qXcHY-Xn9mekHKd1m89sgnW04iUewNw5dGaM31tXFPe4t7ZGcxV803uTaBnBdQ12eBbP2ESI8t8wMCzD0Oejc7roNwnK_oaQFttZmPSxBWSbxibOle_qF-kQISyIu1-q7B00hhfwBp7A791qJpn8nfeFNWvDtOWsUVXn5hp0_RUo5aQCxSO0b-AiUR8Olg4CxF37aBnBVJsdfmoMZ7PgMTpGec-cvHKKni1Sd1h1cCj90KbHE7uPEocg",
                                        .SecretPos = {
                                                .From =   18,
                                                .Len = 430,
                                        },
                                        .MaskPos = {
                                                .From = 106,
                                                .Len = 342,
                                        }},
                        }},
            },
        };

        TSnooper snooper;
        auto searcher = snooper.Searcher(NSecret::ESecretType::TVMTicket);
        for (auto&& [name, testCase] : cases) {
            auto results = searcher->Search(testCase.Content);
            SecretsEqual(testCase.ExpectedSecrets, results, name);
            if (testCase.MaskedContent) {
                TString originalContent = testCase.Content.copy();
                TString result = testCase.Content;
                auto masked = searcher->Mask(result);
                UNIT_ASSERT_C(masked, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(testCase.MaskedContent, result, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(originalContent, testCase.Content, "test " << name);
            }
        }
    }

    Y_UNIT_TEST(TestS3Presign) {
        THashMap<TString, TSimpleCase> cases = {
            {
                "s3_presign",
                TSimpleCase{
                    .Content = R"(response-content-disposition=inline&response-content-type=application%2Fpdf&X-Amz-Content-Sha256=UNSIGNED-PAYLOAD&X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ffopfMqlKkU1fXiJHexn%2F20210721%2Fru-central1%2Fs3%2Faws4_request&X-Amz-Date=20210721T084011Z&X-Amz-SignedHeaders=host&X-Amz-Expires=600&X-Amz-Signature=b98dfcaf857fa912636d8e7af81b426eb70299de2f69aabb0a38561642d204cb)",
                    .MaskedContent = R"(response-content-disposition=inline&response-content-type=application%2Fpdf&X-Amz-Content-Sha256=UNSIGNED-PAYLOAD&X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=ffopfMqlKkU1fXiJHexn%2F20210721%2Fru-central1%2Fs3%2Faws4_request&X-Amz-Date=20210721T084011Z&X-Amz-SignedHeaders=host&X-Amz-Expires=600&X-Amz-Signature=XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX)",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::S3Presign,
                            .Secret = "b98dfcaf857fa912636d8e7af81b426eb70299de2f69aabb0a38561642d204cb",
                            .SecretPos = {
                                .From =   317,
                                .Len = 64,
                            },
                            .MaskPos = {
                                .From = 317,
                                .Len = 64,
                            }},
                    }},
            },
        };

        TSnooper snooper;
        auto searcher = snooper.Searcher(NSecret::ESecretType::S3Presign);
        for (auto&& [name, testCase] : cases) {
            auto results = searcher->Search(testCase.Content);
            SecretsEqual(testCase.ExpectedSecrets, results, name);
            if (testCase.MaskedContent) {
                TString originalContent = testCase.Content.copy();
                TString result = testCase.Content;
                auto masked = searcher->Mask(result);
                UNIT_ASSERT_C(masked, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(testCase.MaskedContent, result, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(originalContent, testCase.Content, "test " << name);
            }
        }
    }

    Y_UNIT_TEST(TestMdsSign) {
        THashMap<TString, TSimpleCase> cases = {
            {
                "mds_sign",
                TSimpleCase{
                    .Content = R"(dynamic-watermark=somesecuretext&ts=1582160381&sign=48b35e4d9f49b7896b46ec7e9896605086e8ad7a71a25c6b39cca33d05dee635)",
                    .MaskedContent = R"(dynamic-watermark=somesecuretext&ts=1582160381&sign=XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX)",
                    .ExpectedSecrets = {
                        TSecret{
                            .Type = ESecretType::MdsSign,
                            .Secret = "48b35e4d9f49b7896b46ec7e9896605086e8ad7a71a25c6b39cca33d05dee635",
                            .SecretPos = {
                                .From =   52,
                                .Len = 64,
                            },
                            .MaskPos = {
                                .From = 52,
                                .Len = 64,
                            }},
                    }},
            },
        };

        TSnooper snooper;
        auto searcher = snooper.Searcher(NSecret::ESecretType::MdsSign);
        for (auto&& [name, testCase] : cases) {
            auto results = searcher->Search(testCase.Content);
            SecretsEqual(testCase.ExpectedSecrets, results, name);
            if (testCase.MaskedContent) {
                TString originalContent = testCase.Content.copy();
                TString result = testCase.Content;
                auto masked = searcher->Mask(result);
                UNIT_ASSERT_C(masked, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(testCase.MaskedContent, result, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(originalContent, testCase.Content, "test " << name);
            }
        }
    }

    Y_UNIT_TEST(TestYC) {
        THashMap<TString, TSimpleCase> cases = {
            {
            "token",
            TSimpleCase{
                .Content = R"(t1.9euelZqbyZGVlM7Pjc3HiY2Wx5zPy-3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_dzEAtv-e9AMD8m_N3z9zM_CG_570AwPyb8.jjLcPfMXhc76214zDpqYQm1b8P37n7cP8PdX5iS7s1aq_C2ZLV3Lhl5JKb5JDUgR9gPIzVkWrHWq7TNEK_xiAw)",
                .MaskedContent = R"(t1.9euelZqbyZGVlM7Pjc3HiY2Wx5zPy-3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_dzEAtv-e9AMD8m_N3z9zM_CG_570AwPyb8.XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX)",
                .ExpectedSecrets = {
                    TSecret{
                        .Type = ESecretType::YCToken,
                        .Secret = R"(t1.9euelZqbyZGVlM7Pjc3HiY2Wx5zPy-3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_dzEAtv-e9AMD8m_N3z9zM_CG_570AwPyb8.jjLcPfMXhc76214zDpqYQm1b8P37n7cP8PdX5iS7s1aq_C2ZLV3Lhl5JKb5JDUgR9gPIzVkWrHWq7TNEK_xiAw)",
                        .SecretPos = {
                                .From = 0,
                                .Len = 186,
                        },
                        .MaskPos = {
                                .From = 100,
                                .Len = 86,
                        }
                    },
                }},
            },
            {
            "cookie",
            TSimpleCase{
                .Content = R"(c1.9euelZqVnc3Gj5mZxs-anZeTksmLmu3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_cjAVBv-e9AJAgj_d3z9yNnMW7570AkCCP91eX174ac0ZCeiouX0ZyQkYyQk5rt-ZCPmpGWm83X9dvIz52czJnIx9LNx8_J0suZm8rSnc6bndLKzMvMyMnInMiem87v_g.m2UCIwIELq-Xp4jab53-yFh2bIohh5uGkGeReWR3KjdEbzgy1b6JTfnMs1TJ1b1w9aJxrstTvDW6XF-mGEF_Cg)",
                .MaskedContent = R"(c1.9euelZqVnc3Gj5mZxs-anZeTksmLmu3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_cjAVBv-e9AJAgj_d3z9yNnMW7570AkCCP91eX174ac0ZCeiouX0ZyQkYyQk5rt-ZCPmpGWm83X9dvIz52czJnIx9LNx8_J0suZm8rSnc6bndLKzMvMyMnInMiem87v_g.XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX)",
                .ExpectedSecrets = {
                    TSecret{
                        .Type = ESecretType::YCCookie,
                        .Secret = R"(c1.9euelZqVnc3Gj5mZxs-anZeTksmLmu3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_cjAVBv-e9AJAgj_d3z9yNnMW7570AkCCP91eX174ac0ZCeiouX0ZyQkYyQk5rt-ZCPmpGWm83X9dvIz52czJnIx9LNx8_J0suZm8rSnc6bndLKzMvMyMnInMiem87v_g.m2UCIwIELq-Xp4jab53-yFh2bIohh5uGkGeReWR3KjdEbzgy1b6JTfnMs1TJ1b1w9aJxrstTvDW6XF-mGEF_Cg)",
                        .SecretPos = {
                            .From = 0,
                            .Len = 280,
                        },
                        .MaskPos = {
                            .From = 194,
                            .Len = 86,
                        }
                    },
                }},
            },
            {
            "apikey",
            TSimpleCase{
                .Content = R"(AQVN0x35bd5AgbWb6X4WpwEW0_2f4sYy5aH5Z_aM)",
                .MaskedContent = R"(AQVNXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX)",
                .ExpectedSecrets = {
                    TSecret{
                        .Type = ESecretType::YCApiKey,
                        .Secret = R"(AQVN0x35bd5AgbWb6X4WpwEW0_2f4sYy5aH5Z_aM)",
                        .SecretPos = {
                            .From = 0,
                            .Len = 40,
                        },
                        .MaskPos = {
                            .From = 4,
                            .Len = 36,
                        }
                    },
                }},
            },
            {
            "static_cred",
            TSimpleCase{
                .Content = R"(YCOFcu7iDT8zC-p0fGkOfEGaaWzHY6TFe3t3av43)",
                .MaskedContent = R"(YCXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX)",
                .ExpectedSecrets = {
                    TSecret{
                        .Type = ESecretType::YCStaticCred,
                        .Secret = R"(YCOFcu7iDT8zC-p0fGkOfEGaaWzHY6TFe3t3av43)",
                        .SecretPos = {
                            .From = 0,
                            .Len = 40,
                        },
                        .MaskPos = {
                            .From = 2,
                            .Len = 38,
                        }
                    },
                }},
            },
        };

        auto secrets = NSecret::ESecretType::YCToken |
                NSecret::ESecretType::YCCookie |
                NSecret::ESecretType::YCApiKey |
                NSecret::ESecretType::YCStaticCred;

        TSnooper snooper;
        auto searcher = snooper.Searcher(secrets);
        for (auto&& [name, testCase] : cases) {
            auto results = searcher->Search(testCase.Content);
            SecretsEqual(testCase.ExpectedSecrets, results, name);
            if (testCase.MaskedContent) {
                TString originalContent = testCase.Content.copy();
                TString result = testCase.Content;
                auto masked = searcher->Mask(result);
                UNIT_ASSERT_C(masked, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(testCase.MaskedContent, result, "test " << name);
                UNIT_ASSERT_VALUES_EQUAL_C(originalContent, testCase.Content, "test " << name);
            }
        }
    }

    Y_UNIT_TEST(TestAll) {
        TString content = R"(
GET /test?sessionid=3%3A1644833359.5.3.1608406008153%3ABAABAAAAAAAFdYCwuAYCKg%3A88.1.2%3A3%7C1120000000038691.35385068.2002.2%3A35385068%7C1120000000102258.393797.2002.0%3A3.2%3A393797%7C1120000000296868.1973956.2402.2%3A1973956%7C1120000000243319.11204194.2002.2%3A11204194%7C1120000000049208.24339104.2002.2%3A24339104%7C1120000000407792.19682120.2002.2%3A19682120%7C1120000000117455.31674725.2002.2%3A31674725%7C1120000000036359.31674875.2002.2%3A31674875%7C5%3A186145.647660.bikybT6drnE-6bzoIJMI_1111_M HTTP/1.1
Host: localhost:9000
User-Agent: HTTPie/1.0.2 AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3
Accept-Encoding: gzip, deflate
Accept: */*
Connection: keep-alive
X-Ya-Service-Ticket: 3:serv:CBAQ__________9_IgQIehAD:Jp93z7FrOT2IOaPtAD3s7_zFU6DJrL3HvHbYJKu9Cet5ytYSKzseN-SVmIMH5y4UM51XtJcV3mvD1rv6w_dzt36PxpS7R7EAPbQWkP6wJGQRdKWjmAWkkqD-s062kL4q39fG0hJwxe4s-8zDh5hCXTimzZvFrdo09hQcfFAoj-cJp93z7FrOT2IOaPtAD3s7_zFU6DJrL3HvHbYJKu9Cet5ytYSKzseN-SVmIMH5y4UM51XtJcV3mvD1rv6w_dzt36PxpS7R7EAPbQWkP6wJGQRdKWjmAWkkqD-s062kL4q39fG0hJwxe4s-8zDh5hCXTimzZvFrdo09hQcfFAoj-c
X-YC-Key: AQVN0x35bd5AgbWb6X4WpwEW0_2f4sYy5aH5Z_aM
Cookie: yandexuid=654321; Session_id=3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.ySsdPJeQSM1Zoj32qOMvTKSWVxM; sessionid2=3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.OJbvg_ahfMoFX22xPvf-iEoVNsk; something=brown; yc_session=c1.9euelZqVnc3Gj5mZxs-anZeTksmLmu3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_cjAVBv-e9AJAgj_d3z9yNnMW7570AkCCP91eX174ac0ZCeiouX0ZyQkYyQk5rt-ZCPmpGWm83X9dvIz52czJnIx9LNx8_J0suZm8rSnc6bndLKzMvMyMnInMiem87v_g.m2UCIwIELq-ApFjab5J-yFh5bIohh5uAkGeReWR3KjdEbzgy5b6JTfnAs1TJ1b1w9aJxrstTvDW6XF-mGEF_Cg;
Authorization: OAuth AgAAAAAHYVLuAAX_m477zavGV0GMlcItT-kpRm3
Authorization-V3: OAuth y0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSA

)";

        TString maskedContent = R"(
GET /test?sessionid=3%3A1644833359.5.3.1608406008153%3ABAABAAAAAAAFdYCwuAYCKg%3A88.1.2%3A3%7C1120000000038691.35385068.2002.2%3A35385068%7C1120000000102258.393797.2002.0%3A3.2%3A393797%7C1120000000296868.1973956.2402.2%3A1973956%7C1120000000243319.11204194.2002.2%3A11204194%7C1120000000049208.24339104.2002.2%3A24339104%7C1120000000407792.19682120.2002.2%3A19682120%7C1120000000117455.31674725.2002.2%3A31674725%7C1120000000036359.31674875.2002.2%3A31674875%7C5%3A186145.647660.XXXXXXXXXXXXXXXXXXXXXXXXXXX HTTP/1.1
Host: localhost:9000
User-Agent: HTTPie/1.0.2 AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX
Accept-Encoding: gzip, deflate
Accept: */*
Connection: keep-alive
X-Ya-Service-Ticket: 3:serv:CBAQ__________9_IgQIehAD:XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
X-YC-Key: AQVNXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
Cookie: yandexuid=654321; Session_id=3:1566480756.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.490331.2002.2:490331|142241.804415.XXXXXXXXXXXXXXXXXXXXXXXXXXX; sessionid2=3:1566820123.5.0.1564091077870:YWgJJQ:88.1|1120000000038691.0.2002|1120000000049208.-1.0|142429.844023.XXXXXXXXXXXXXXXXXXXXXXXXXXX; something=brown; yc_session=c1.9euelZqVnc3Gj5mZxs-anZeTksmLmu3rnpWalI3Pi4uQm5iai4ydkpCWnpTl8_cjAVBv-e9AJAgj_d3z9yNnMW7570AkCCP91eX174ac0ZCeiouX0ZyQkYyQk5rt-ZCPmpGWm83X9dvIz52czJnIx9LNx8_J0suZm8rSnc6bndLKzMvMyMnInMiem87v_g.XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX;
Authorization: OAuth AgAAAAAHYVLuAAX_m4XXXXXXXXXXXXXXXXXXXXX
Authorization-V3: OAuth y0_AQAAAAAAAYiUAAAAgQAAAAA63mixXXXXXXXXXXXXXXXXXXXXXXXXXXX

)";

        TSnooper snooper;
        auto searcher = snooper.Searcher(NSecret::ESecretType::All);
        auto searchResults = searcher->Search(content);
        UNIT_ASSERT_VALUES_EQUAL(9, searchResults.size());
        auto maskResults = searcher->Mask(content);
        UNIT_ASSERT_VALUES_EQUAL(9, maskResults.size());
        UNIT_ASSERT_VALUES_EQUAL(maskedContent, content);
    }
}
