#include "searcher.h"

#include <security/ant-secret/snooper/internal/searchers/all.h>

#include <util/generic/strbuf.h>
#include <util/generic/set.h>

namespace NSnooperInt {
    namespace {
        using TSpottedMap = THashMap<std::pair<size_t, size_t>, TSet<std::pair<size_t, size_t>, std::less<>>>;
    }

    TSearcher::TSearcher(TContext& ctx, TSearcherOpts opts)
        : ctx(ctx)
        , opts(opts)
    {
        // Order matters!!!
        if (opts.SecretTypes & NSecret::ESecretType::YSession) {
            searchers.push_back(MakeHolder<NSearchers::TYandexSession>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::YOAuth) {
            searchers.push_back(MakeHolder<NSearchers::TYandexOAuth>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::TVMTicket) {
            searchers.push_back(MakeHolder<NSearchers::TTvmServiceTicket>(ctx));
            searchers.push_back(MakeHolder<NSearchers::TTvmUserTicket>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::S3Presign) {
            searchers.push_back(MakeHolder<NSearchers::TS3Presign>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::MdsSign) {
            searchers.push_back(MakeHolder<NSearchers::TMdsSign>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::YCApiKey) {
            searchers.push_back(MakeHolder<NSearchers::TYCApiKey>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::YCCookie) {
            searchers.push_back(MakeHolder<NSearchers::TYCCookie>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::YCToken) {
            searchers.push_back(MakeHolder<NSearchers::TYCToken>(ctx));
        }

        if (opts.SecretTypes & NSecret::ESecretType::YCStaticCred) {
            searchers.push_back(MakeHolder<NSearchers::TYCStaticCred>(ctx));
        }

        {
            TVector<const char*> expressions;
            TVector<unsigned int> ids;
            TVector<unsigned int> flags;

            unsigned int id = 0;
            const unsigned int flag = HS_FLAG_DOTALL | HS_FLAG_SOM_LEFTMOST;
            for (size_t searcherId = 0; searcherId < searchers.size(); ++searcherId) {
                const auto& patterns = searchers[searcherId]->KeyPatterns();
                for (size_t patternId = 0; patternId < patterns.size(); ++patternId) {
                    id++;
                    keyToSearcher[id] = {searcherId, patternId};
                    expressions.push_back(patterns[patternId].data());
                    ids.push_back(id);
                    flags.push_back(flag);
                }
            }

            keysRe = NSSInternal::TRegexStorage::Instance().GetOrCompile("keys_db", expressions, flags, ids);
        }
    }

    NSecret::TSecretList TSearcher::Search(TStringBuf data, bool validOnly) {
        TSpottedMap spottedSearchers;
        auto matcher = [&spottedSearchers, this](unsigned int id, unsigned long long from,
                                                       unsigned long long to) {
            spottedSearchers[std::make_pair<size_t, size_t>(from, to)].insert(keyToSearcher[id]);
        };

        NHyperscan::ScanPtr(keysRe.db, keysRe.scratch, data, matcher);

        NSecret::TSecretList secrets;
        for (auto&& [pos, searcherIDs] : spottedSearchers) {
            for (const auto& searcherID : searcherIDs) {
                TMaybe<NSecret::TSecret> secret;
                NSearchers::TSearchRequest req {
                    .data = data,
                    .keyFrom = pos.first,
                    .keyTo = pos.second,
                    .keyId = searcherID.second,
                };

                if (validOnly) {
                    secret = searchers[searcherID.first]->SearchValidated(req);
                } else {
                    secret = searchers[searcherID.first]->Search(req);
                }

                if (!secret.Defined()) {
                    continue;
                }

                secrets.push_back(secret.GetRef());
                break;
            }
        }

        return secrets;
    }

    NSecret::TSecretList TSearcher::Mask(TString& data, bool validOnly) {
        auto secrets = Search(data, validOnly);
        if (!secrets) {
            return {};
        }
        MaskAtSecrets(data, secrets);
        return secrets;
    }

    void TSearcher::CollectRegexes() {
        // TODO(buglloc): implement me, huh?
    }

}
