#include "prefixed.h"

#include <util/string/vector.h>

#include <util/string/join.h>

namespace NSearchers {
    namespace {
        const TVector<TString> kKvSeparators{
            R"( )",
            R"(\s*: )",
            R"(\\?=)",
        };

        const TVector<TString> kBounds{
            R"(\s*:\s*)",
            R"(\s*%3[Aa]\s*)",
            R"((?:\s|\\)*=(?:\s|\\)*)",
            R"((?:\s|\\)*=>(?:\s|\\)*)",
        };

        const TVector<TString> kQuotes{
            R"(\\*")",
            R"(\\*%22)",
            R"(\\*')",
        };

        const TString kAnchors = "\\'\"\x20\t\r\n\f\v&#;";
        const TString kEncQuote = "%22";

        inline size_t findAnchor(const TStringBuf data, size_t from) {
            auto anchorPos = data.find_first_of(kAnchors, from);
            if (anchorPos != TString::npos) {
                return anchorPos - from;
            }

            return data.size() - from;
        }

        inline size_t findEncQuote(const TStringBuf data, size_t from) {
            auto anchorPos = data.find(kEncQuote, from);
            if (anchorPos != TString::npos) {
                return anchorPos - from;
            }

            return data.size() - from;
        }

        inline TVector<TString> genQuotedSeparator() {
            TVector<TString> out;
            for (auto&& quote : kQuotes) {
                for (auto&& sep : kBounds) {
                    out.push_back(quote+sep+quote);
                }
            }
            return out;
        }
    }

    TPrefixed::TPrefixed(NSnooperInt::TContext& ctx, const TVector<TString>& keys, const TVector<TString>& values)
        : ctx(ctx)
    {
        auto quotedSeparators = genQuotedSeparator();

        TVector<TString> separators;
        separators.insert(separators.end(), kKvSeparators.begin(), kKvSeparators.end());
        separators.insert(separators.end(), quotedSeparators.begin(), quotedSeparators.end());
        TString kvSeparator = "(?:" + JoinSeq("|", separators) + ")";
        for (auto&& pattern : keys) {
            keyPatterns.push_back("(?i:" + pattern + ")" + kvSeparator);
        }

        TVector<TString> valuePatterns;
        for (auto&& pattern : values) {
            valuePatterns.push_back("^" + pattern + "$");
        }

        re = CompilePatterns("prefixed_searcher_db", valuePatterns, HS_FLAG_DOTALL | HS_FLAG_SINGLEMATCH);
    }

    TVector<TString> TPrefixed::KeyPatterns() const {
        return keyPatterns;
    }

    TMaybe<NSecret::TSecret> TPrefixed::Search(const TSearchRequest& req) {
        auto from = req.keyTo;
        if (req.data.length() <= from) {
            return {};
        }
        
        size_t len = 0;
        if (from > 3) {
                if (req.data[from-3] == kEncQuote[0] && req.data[from-2] == kEncQuote[1] && req.data[from-1] == kEncQuote[2]) {
                        len = findEncQuote(req.data, from);
                }
        }

        if (len == 0) { 
                len = findAnchor(req.data, from);
        }
        auto rawSecret = req.data.SubString(from, len);

        TString secret = decodeSecret(rawSecret);
        TMaybe<NSecret::TSecret> result;
        auto matcher = [&result, secret, rawSecret, from, len, this](unsigned int id, unsigned long long /*from*/,
                                                          unsigned long long /*to*/) {
            if (IsSecret(id, secret)) {
                auto mask = MaskSecret(id, rawSecret);
                // add initial position to the mask info
                mask.From += from;

                result = NSecret::TSecret{
                    .Type = SecretType(),
                    .Secret = secret,
                    .SecretPos = NSecret::TPos{
                        .From = from,
                        .Len = len,
                    },
                    .MaskPos = mask,
                };
            }
        };

        NHyperscan::ScanPtr(re.db, re.scratch, secret, matcher);
        return result;
    }

    TMaybe<NSecret::TSecret> TPrefixed::SearchValidated(const TSearchRequest& req) {
        auto secret = Search(req);
        if (!ctx.Validator || !secret) {
            return Nothing();
        }

        TMaybe<bool> forced = ForceValid();
        if (forced.Defined()) {
            return forced.GetRef() ? secret : Nothing();
        }

        const auto& info = ctx.Validator->Call(Name(), secret->Secret);
        if (!info || !info->Valid) {
            return Nothing();
        }

        return secret;
    }

}
