#include "tvm_service.h"

#include <security/ant-secret/internal/string_utils/common.h>

#include <util/string/vector.h>


namespace NSearchers {
    namespace {
        const TVector<TString> kValuePatterns{
            R"(3\:serv\:[\w\-]+\:[\w\-]{342})",
        };

        bool isServiceTicket(const TStringBuf rawTicket) {
            TStringBuf ticket;
            if (!rawTicket.AfterPrefix("3:serv:", ticket)) {
                return false;
            }

            TStringBuf sign = ticket.After(':');
            if (NStringUtils::IsMasked(sign, 100)) {
                return false;
            }

            TStringBuf body = ticket.Before(':');
            return NStringUtils::IsBase64Url(body) && NStringUtils::IsBase64Url(sign);
        }
    }

    TTvmServiceTicket::TTvmServiceTicket(NSnooperInt::TContext& ctx)
        : TWhole(ctx, kValuePatterns)
    {
    }

    bool TTvmServiceTicket::IsSecret(size_t id, const TStringBuf secret) {
        Y_UNUSED(id);
        return isServiceTicket(secret);
    }

    NSecret::TPos TTvmServiceTicket::MaskSecret(size_t id, const TStringBuf rawSecret) {
        Y_UNUSED(id);

        return {
            .From = rawSecret.length() - 342,
            .Len = 342,
        };
    }

    TString TTvmServiceTicket::Name() const {
        return "tvm-service-ticket";
    }

    NSecret::ESecretType TTvmServiceTicket::SecretType() const {
        return NSecret::ESecretType::TVMTicket;
    }

    TMaybe<bool> TTvmServiceTicket::ForceValid() const {
        return true;
    }
}
