#include "whole.h"

#include <util/string/vector.h>

#include <util/string/join.h>
#include <util/string/subst.h>
#include <library/cpp/string_utils/quote/quote.h>

namespace NSearchers {
    namespace {
        const TVector<std::pair<TString, TString>> kBounds{
                {R"(^)", R"((?:[\s\\]|$))"},
                {R"(\s)", R"((?:[\s\\'"`;]|$))"},
                {R"((?:%20|%22))", R"((?:%22|%20))"}, // suitable for urlencoded json
                {R"(')", R"(['\\])"},
                {R"(`)", R"([`\\])"},
                {R"(")", R"(["\\])"},
                {R"(=)", R"((?:[&'"`#\\\s;]|$))"},
                {R"(#)", R"((?:[&'"`\\\s]|$))"},
        };

        inline std::pair<size_t, size_t> secretPos(const TStringBuf data, size_t from, size_t to) {
            if (data[from] == '%') {
                from += 3;
                to -= 3; 
            } else {
                    if (!std::isalnum(data[from])) {
                        from++;
                    }

                    if (!std::isalnum(data[to - 1])) {
                        to--;
                    }
            }
            return {from, to - from};
        }

        TString urlifyPattern(const TString& pattern) {
            auto out = SubstGlobalCopy(pattern, "\\:", "(:|%3[Aa])");
            SubstGlobal(out, "\\|", "(\\||%7[Cc])");
            SubstGlobal(out, "\\.", "(\\.|%2[Ee])");
            return out;
        }
    }

    TWhole::TWhole(NSnooperInt::TContext &ctx, const TVector<TString> &values)
            : ctx(ctx) {

        for (auto &&value : values) {
            auto pattern = urlifyPattern(value);
            for (auto&&[left, right] : kBounds) {
                patterns.push_back(left + pattern + right);
            }
        }
    }

    TVector<TString> TWhole::KeyPatterns() const {
        return patterns;
    }

    TMaybe<NSecret::TSecret> TWhole::Search(const TSearchRequest &req) {
        auto [from, len] = secretPos(req.data, req.keyFrom, req.keyTo);
        auto rawSecret = req.data.SubString(from, len);
        TString secret = decodeSecret(rawSecret);
        size_t patternID = req.keyId / kBounds.size();
        if (!IsSecret(patternID, secret)) {
            return {};
        }

        auto mask = MaskSecret(patternID, rawSecret);
        // add initial position to the mask info
        mask.From += from;

        return NSecret::TSecret{
                .Type = SecretType(),
                .Secret = secret,
                .SecretPos = NSecret::TPos{
                    .From = from,
                    .Len = len,
                },
                .MaskPos = mask,
        };
    }

    TMaybe<NSecret::TSecret> TWhole::SearchValidated(const TSearchRequest &req) {
        auto secret = Search(req);
        if (!ctx.Validator || !secret) {
            return Nothing();
        }

        TMaybe<bool> forced = ForceValid();
        if (forced.Defined()) {
            return forced.GetRef() ? secret : Nothing();
        }

        const auto &info = ctx.Validator->Call(Name(), secret->Secret);
        if (!info || !info->Valid) {
            return Nothing();
        }

        return secret;
    }

}
