#include "yandex_oauth.h"

#include <security/ant-secret/internal/blackbox/oauth.h>


namespace NSearchers {
    namespace {
/*
We support only 2 types of oauth tokens:
1. token with embedded info - base64url(uid, clid, shard, random), [a-zA-Z0-9_-]+, e.g.:
    AQAD-qJSJpcjAAADwHcUZ2Serk5EmBjzGpemfJQ
2. stateless token - ver.uid.client_id.expires.token_id.key_id.iv[12 bytes].data.tag[16 bytes], e.g.:
    1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q
3. new token with embedded info : y(env)_base64url(shard, uid, clientid, token_id, random, crc32), e.g.:
    y0_AQAAAAAAAYiUAAAAgQAAAAA63mixeHh4eHh4eHh4eHh4eHh4eMmOnSA

Old (MD5) tokens are excluded.
*/
        const TVector<TString> kValuePatterns{
            // embedded - remove prefix?
            R"(A[A-Za-z0-9_\-]{37}[A-Za-z0-9])",
            // new embed
            R"(y[0-9]_[A-Za-z0-9_\-]{55})",
            // stateless, for now version is always 1
            R"(1\.\d+\.\d+\.\d+\.\d+\.\d+\.[\w\-]{16}\.[\w\-]+\.[\w\-]{22})",
        };

    }

    TYandexOAuth::TYandexOAuth(NSnooperInt::TContext& ctx)
        : TWhole(ctx, kValuePatterns)
    {
    }

    bool TYandexOAuth::IsSecret(size_t id, const TStringBuf token) {
        switch (id) {
            case 0:
                // embedded
                return NBlackBox::IsEmbeddedV2Token(token);
            case 1:
                // new embedded
                return NBlackBox::IsEmbeddedV3Token(token);
            case 2:
                // stateless
                // passport/infra/daemons/blackbox/src/oauth_processor.cpp:
                // token.size() > 70 && NPassport::NUtils::beginsWith(token, "1.")
                return token.size() > 70;
            default:
                ythrow TSystemError() << "unexpected token pattern id: " << id;
        }
    }

    NSecret::TPos TYandexOAuth::MaskSecret(size_t id, const TStringBuf rawSecret) {
        switch (id) {
            case 0:
                // we left shard + uid + clientid
                return {
                    .From = 18,
                    .Len = 21,
                };
            case 1:
                // we left prefix + shard + uid + clientid + tokenid
                return {
                    .From = 31,
                    .Len = 27,
                };
            case 2:
                return {
                    .From = rawSecret.length() - 22,
                    .Len = 22,
                };
            default:
                ythrow TSystemError() << "unexpected token pattern id: " << id;
        }
    }

    TString TYandexOAuth::Name() const {
        return "yandex-oauth";
    }

    NSecret::ESecretType TYandexOAuth::SecretType() const {
        return NSecret::ESecretType::YOAuth;
    }

    TMaybe<bool> TYandexOAuth::ForceValid() const {
        return Nothing();
    }
}
