from util.generic.string cimport TString, TStringBuf
from util.generic.hash cimport THashMap
from util.generic.vector cimport TVector
from libcpp cimport bool


cdef extern from "security/ant-secret/snooper/internal/secret/secret_types.h" namespace "NSecret":
    cdef enum ESecretType:
        Unknown 'NSecret::ESecretType::Unknown',
        YOAuth 'NSecret::ESecretType::YOAuth',
        YSession 'NSecret::ESecretType::YSession',
        TVMTicket 'NSecret::ESecretType::TVMTicket',
        S3Presign 'NSecret::ESecretType::S3Presign',
        MdsSign 'NSecret::ESecretType::MdsSign',
        YCApiKey 'NSecret::ESecretType::YCApiKey',
        YCCookie 'NSecret::ESecretType::YCCookie',
        YCToken 'NSecret::ESecretType::YCToken',
        YCStaticCred 'NSecret::ESecretType::YCStaticCred',
        All 'NSecret::ESecretType::All'
        AllWMds 'NSecret::ESecretType::AllWMds'

cdef extern from "security/ant-secret/snooper/cpp/secret.h" namespace "NSnooper":
    cdef cppclass TPos:
        size_t From
        size_t Len

    cdef cppclass TSecret:
        ESecretType Type
        TString Secret
        TPos SecretPos
        TPos MaskPos

    ctypedef TVector[TSecret] TSecretList


cdef extern from "security/ant-secret/snooper/cpp/searcher.h" namespace "NSnooper":
    cdef cppclass TSearcher:
        TSecretList Search(TStringBuf, bool)
        bool Mask(TString&, bool)


cdef extern from "security/ant-secret/snooper/cpp/snooper.h" namespace "NSnooper":
    cdef cppclass TSnooper:
        TSearcher* NewSearcher()
        TSearcher Searcher()


class SecretType(object):
    def __init__(self, s_type, s_name):
        self.type = s_type
        self.name = s_name

    def __repr__(self):
         return "SecretType(%r, %r)" % (self.type, self.name)

    def __str__(self):
         return self.name


cdef class SecretTypes(object):
    unknown = SecretType(ESecretType.Unknown, "unknown")
    yandex_oauth = SecretType(ESecretType.YOAuth, "yandex_oauth")
    yandex_session = SecretType(ESecretType.YSession, "yandex_session")
    tvm_ticket = SecretType(ESecretType.TVMTicket, "tvm_ticket")
    s3_presign = SecretType(ESecretType.S3Presign, "s3_presign")
    mds_sign = SecretType(ESecretType.MdsSign, "mds_sign")
    yc_api_key = SecretType(ESecretType.YCApiKey, "yc_api_key")
    yc_cookie = SecretType(ESecretType.YCCookie, "yc_cookie")
    yc_token = SecretType(ESecretType.YCToken, "yc_token")
    yc_static_cred = SecretType(ESecretType.YCStaticCred, "yc_static_cred")
    all = SecretType(ESecretType.All, "all")
    all_w_mds = SecretType(ESecretType.AllWMds, "all_w_mds")

    @staticmethod
    cdef _factory(ESecretType val):
        if val == ESecretType.YOAuth:
            return SecretTypes.yandex_oauth
        elif val == ESecretType.YSession:
            return SecretTypes.yandex_session
        elif val == ESecretType.TVMTicket:
            return SecretTypes.tvm_ticket
        elif val == ESecretType.S3Presign:
            return SecretTypes.s3_presign
        elif val == ESecretType.MdsSign:
            return SecretTypes.mds_sign
        elif val == ESecretType.YCApiKey:
            return SecretTypes.yc_api_key
        elif val == ESecretType.YCCookie:
            return SecretTypes.yc_cookie
        elif val == ESecretType.YCToken:
            return SecretTypes.yc_token
        elif val == ESecretType.YCStaticCred:
            return SecretTypes.yc_static_cred
        else:
            raise Exception('unknown secret type: %r' % val)


class Secret(object):
    __slots__ = ('type', 'secret', 'secret_from', 'secret_to', 'secret_len', 'mask_from', 'mask_to', 'mask_len')

    def __init__(self, type, secret, secret_from, secret_len, mask_from, mask_len):
        self.type = type
        self.secret = secret
        self.secret_from = secret_from
        self.secret_to = self.secret_from + secret_len
        self.secret_len = secret_len
        self.mask_from = mask_from
        self.mask_to = self.mask_from + mask_len
        self.mask_len = mask_len

    def to_dict(self):
        return dict(
            type=str(self.type),
            secret=self.secret,
            secret_pos=self.secret_pos,
            mask_pos=self.mask_pos,
        )

    @property
    def mask_pos(self):
        return self.mask_from, self.mask_to

    @property
    def secret_pos(self):
        return self.secret_from, self.secret_to

    @property
    def is_yandex_oauth(self):
        return self.type == SecretTypes.yandex_oauth

    @property
    def is_yandex_session(self):
        return self.type == SecretTypes.yandex_session

    @property
    def is_tvm_ticket(self):
        return self.type == SecretTypes.tvm_ticket

    @property
    def is_s3_presign(self):
        return self.type == SecretTypes.s3_presign

    @property
    def is_mds_sign(self):
        return self.type == SecretTypes.mds_sign

    @property
    def is_yc_api_key(self):
        return self.type == SecretTypes.yc_api_key

    @property
    def is_yc_cookie(self):
        return self.type == SecretTypes.yc_cookie

    @property
    def is_yc_token(self):
        return self.type == SecretTypes.yc_token

    @property
    def is_yc_static_cred(self):
        return self.type == SecretTypes.yc_static_cred


cdef class Snooper(object):
    cdef TSnooper* snooper

    def __cinit__(self):
        self.snooper = new TSnooper()

    def __dealloc__(self):
        if self.snooper is not NULL:
            del self.snooper
            self.snooper = NULL

    def searcher(self, secret_types = [SecretTypes.all]):
        return Searcher._from_ptr(self.snooper.NewSearcher())


cdef class Searcher(object):
    cdef TSearcher* searcher

    def __cinit__(self):
        self.searcher = NULL

    def __dealloc__(self):
        if self.searcher is not NULL:
            del self.searcher
            self.searcher = NULL

    @staticmethod
    cdef Searcher _from_ptr(TSearcher* ptr):
        searcher = Searcher()
        searcher.searcher = ptr
        return searcher

    def search(self, TStringBuf content, bool valid_only = False):
        results = self.searcher.Search(content, valid_only)
        for secret in results:
            yield Secret(
                type=SecretTypes._factory(secret.Type),
                secret=secret.Secret,
                secret_from=secret.SecretPos.From,
                secret_len=secret.SecretPos.Len,
                mask_from=secret.MaskPos.From,
                mask_len=secret.MaskPos.Len,
            )

    def mask(self, TString content, bool valid_only = False):
        self.searcher.Mask(content, valid_only)
        return content
