package cache

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
)

type (
	DB struct {
		ctx                           context.Context
		sp                            *table.SessionPool
		pingQuery                     string
		selectStaffUserQuery          string
		selectKnownSecretQuery        string
		replaceKnownSecretQuery       string
		selectKnownUnknownSecretQuery string
		selectUnknownSecretQuery      string
		replaceUnknownSecretQuery     string
		selectSSHKeyQuery             string
		selectCrlQuery                string
		selectTvmInfoQuery            string
	}

	Config struct {
		AuthToken string
		Database  string
		Endpoint  string
	}

	TvmInfo struct {
		ResourceID uint64 `json:"resource_id"`
		Tags       []struct {
			ID   uint64 `json:"id"`
			Name string `json:"name"`
		} `json:"tags"`
	}
)

func NewDB(ctx context.Context, cfg Config) (*DB, error) {
	config := new(ydb.DriverConfig)

	config.Credentials = ydb.AuthTokenCredentials{
		AuthToken: cfg.AuthToken,
	}

	config.Database = cfg.Database

	driver, err := (&ydb.Dialer{
		DriverConfig: config,
	}).Dial(ctx, cfg.Endpoint)

	if err != nil {
		return nil, fmt.Errorf("dial error: %v", err)
	}

	tableClient := table.Client{
		Driver: driver,
	}

	sp := table.SessionPool{
		IdleThreshold: 5 * time.Second,
		Builder:       &tableClient,
	}

	err = createTables(ctx, &sp, cfg.Database)
	if err != nil {
		return nil, fmt.Errorf("create tables error: %v", err)
	}

	return &DB{
		ctx:                           ctx,
		sp:                            &sp,
		pingQuery:                     pingQuery(cfg.Database),
		selectStaffUserQuery:          selectStaffUserQuery(cfg.Database),
		selectCrlQuery:                selectCrlQuery(cfg.Database),
		selectKnownSecretQuery:        selectKnownSecretQuery(cfg.Database),
		selectSSHKeyQuery:             selectSSHKeyQuery(cfg.Database),
		selectTvmInfoQuery:            selectTvmInfoQuery(cfg.Database),
		selectUnknownSecretQuery:      selectUnknownSecretQuery(cfg.Database),
		selectKnownUnknownSecretQuery: selectKnownUnknownSecretQuery(cfg.Database),
		replaceKnownSecretQuery:       replaceKnownSecretQuery(cfg.Database),
		replaceUnknownSecretQuery:     replaceUnknownSecretQuery(cfg.Database),
	}, nil
}

func (db *DB) Close() error {
	return db.sp.Close(db.ctx)
}

func (db *DB) Ping() (err error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.pingQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx,
				table.NewQueryParameters(),
			)
			return
		}),
	)

	if err == nil {
		if !(res.NextSet() && res.NextRow()) {
			err = errors.New("nothing selected, but must")
		}
	}
	return
}

func (db *DB) IsKnown(sha1 string) (known bool, err error) {
	// Prepare read transaction
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.selectKnownSecretQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$sha1", ydb.StringValue([]byte(sha1))),
			))
			return
		}),
	)

	known = err == nil && res.NextSet() && res.NextRow()
	return
}

func (db *DB) TouchKnown(sha1 string, meta interface{}) (err error) {
	// Prepare write transaction.
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	metaJSON, err := json.Marshal(meta)
	if err != nil {
		return fmt.Errorf("failed to marshal meta: %w", err)
	}

	updatedAt := uint64(time.Now().Unix())
	return table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.replaceKnownSecretQuery)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$sha1", ydb.StringValue([]byte(sha1))),
				table.ValueParam("$updated_at", ydb.Uint64Value(updatedAt)),
				table.ValueParam("$meta", ydb.JSONValue(string(metaJSON))),
			))
			return err
		}),
	)
}

func (db *DB) IsUnknown(sha1, validator string) (unknown bool, err error) {
	// Prepare read transaction
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.selectUnknownSecretQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$sha1", ydb.StringValue([]byte(sha1))),
				table.ValueParam("$validator", ydb.StringValue([]byte(validator))),
			))
			return
		}),
	)

	unknown = err == nil && res.NextSet() && res.NextRow()
	return
}

func (db *DB) IsKnownOrUnknown(sha1, validator string) (unknown bool, err error) {
	// Prepare read transaction
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.selectKnownUnknownSecretQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$sha1", ydb.StringValue([]byte(sha1))),
				table.ValueParam("$validator", ydb.StringValue([]byte(validator))),
			))
			return
		}),
	)

	unknown = err == nil && res.NextSet() && res.NextRow()
	return
}

func (db *DB) TouchUnknown(sha1, validator string) (err error) {
	// Prepare write transaction.
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	return table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.replaceUnknownSecretQuery)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$sha1", ydb.StringValue([]byte(sha1))),
				table.ValueParam("$validator", ydb.StringValue([]byte(validator))),
				table.ValueParam("$ttl", ydb.Uint64Value(generateTTL())),
			))
			return err
		}),
	)
}

func (db *DB) IsTLSRevoked(ca, serial string) (revoked bool, err error) {
	// Prepare read transaction
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.selectCrlQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$serial", ydb.StringValue([]byte(serial))),
				table.ValueParam("$ca", ydb.StringValue([]byte(ca))),
			))
			return
		}),
	)

	revoked = err == nil && res.NextSet() && res.NextRow()
	return
}

func (db *DB) SSHKeyLogins(fingerprint string) (logins []string, err error) {
	// Prepare read transaction
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.selectSSHKeyQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$fingerprint", ydb.StringValue([]byte(fingerprint))),
			))
			return
		}),
	)

	if err != nil {
		return
	}

	for res.NextSet() {
		for res.NextRow() {
			res.NextItem()
			logins = append(logins, string(res.OString()))
		}
	}

	err = res.Err()
	return
}

func (db *DB) UIDToLogin(uid uint64) (login string, err error) {
	// Prepare read transaction
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.selectStaffUserQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$uid", ydb.Uint64Value(uid)),
			))
			return
		}),
	)

	if err != nil || !res.NextSet() || !res.NextRow() {
		return
	}

	res.NextItem()
	login = string(res.OString())
	err = res.Err()
	return
}

func (db *DB) TvmInfo(clientID uint64) (info TvmInfo, err error) {
	// Prepare read transaction
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	err = table.Retry(db.ctx, db.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, db.selectTvmInfoQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$tvm_id", ydb.Uint64Value(clientID)),
			))
			return
		}),
	)

	if err != nil || !res.NextSet() || !res.NextRow() {
		return
	}

	// SELECT resource_id, tags
	res.NextItem()
	info.ResourceID = res.OUint64()

	res.NextItem()
	tags := res.OJSON()
	if tags != "" {
		err = json.Unmarshal([]byte(tags), &info.Tags)
		if err != nil {
			return
		}
	}

	err = res.Err()
	return
}
