package cache

import (
	"context"
	"fmt"
	"math/rand"
	"path"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
)

const (
	unknownSecretTTLLow  = 27
	unknownSecretTTLHigh = 40
)

func init() {
	rand.Seed(time.Now().Unix())
}

func pingQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $uid AS Uint64;

SELECT 1
  FROM staff_users
WHERE uid = 1120000000038691
LIMIT 1;
`, pathPrefix)
}

func selectStaffUserQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $uid AS Uint64;

SELECT login
  FROM staff_users
WHERE uid = $uid
LIMIT 1;
`, pathPrefix)
}

func selectKnownSecretQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $sha1 AS String;

SELECT 1
  FROM known_secrets
WHERE sha1 = $sha1
LIMIT 1;
`, pathPrefix)
}

func replaceKnownSecretQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $sha1 AS String;
DECLARE $updated_at AS Uint64;
DECLARE $meta AS Json;

REPLACE INTO known_secrets (sha1, updated_at, meta) VALUES ($sha1, $updated_at, $meta);
`, pathPrefix)
}

func selectUnknownSecretQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $sha1 AS String;
DECLARE $validator AS String;

SELECT 1
  FROM unknown_secrets
WHERE sha1 = $sha1 and validator = $validator
LIMIT 1;
`, pathPrefix)
}

func selectKnownUnknownSecretQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $sha1 AS String;
DECLARE $validator AS String;

SELECT 1 FROM unknown_secrets
WHERE
  sha1 = $sha1 AND validator = $validator
LIMIT 1
UNION ALL
  SELECT 1 FROM known_secrets
  WHERE
    sha1 = $sha1
  LIMIT 1;
`, pathPrefix)
}

func replaceUnknownSecretQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $sha1 AS String;
DECLARE $validator AS String;
DECLARE $ttl AS Uint64;

REPLACE INTO unknown_secrets (sha1, validator, ttl) VALUES ($sha1, $validator, $ttl);
`, pathPrefix)
}

func selectSSHKeyQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $fingerprint AS String;

SELECT login
  FROM ssh_keys
WHERE fingerprint = $fingerprint;
`, pathPrefix)
}

func selectCrlQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $serial AS String;
DECLARE $ca AS String;

SELECT 1
  FROM crls
WHERE serial = $serial AND ca = $ca
LIMIT 1;
`, pathPrefix)
}

func selectTvmInfoQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $tvm_id AS Uint64;

SELECT resource_id, tags
  FROM tvm_tags
WHERE tvm_id = $tvm_id
LIMIT 1;
`, pathPrefix)
}

func createTables(ctx context.Context, sp *table.SessionPool, prefix string) (err error) {
	err = table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, path.Join(prefix, "unknown_secrets"),
				table.WithColumn("sha1", ydb.Optional(ydb.TypeString)),
				table.WithColumn("validator", ydb.Optional(ydb.TypeString)),
				table.WithColumn("ttl", ydb.Optional(ydb.TypeUint64)),
				table.WithPrimaryKeyColumn("sha1", "validator"),
			)
		}),
	)
	if err != nil {
		return
	}

	err = table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, path.Join(prefix, "known_secrets"),
				table.WithColumn("sha1", ydb.Optional(ydb.TypeString)),
				table.WithColumn("updated_at", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("meta", ydb.Optional(ydb.TypeJSON)),
				table.WithPrimaryKeyColumn("sha1"),
			)
		}),
	)
	if err != nil {
		return
	}

	err = table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, path.Join(prefix, "ssh_keys"),
				table.WithColumn("fingerprint", ydb.Optional(ydb.TypeString)),
				table.WithColumn("login", ydb.Optional(ydb.TypeString)),
				table.WithColumn("updated_at", ydb.Optional(ydb.TypeUint64)),
				table.WithPrimaryKeyColumn("fingerprint", "login"),
			)
		}),
	)
	if err != nil {
		return
	}

	err = table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, path.Join(prefix, "crls"),
				table.WithColumn("serial", ydb.Optional(ydb.TypeString)),
				table.WithColumn("ca", ydb.Optional(ydb.TypeString)),
				table.WithColumn("updated_at", ydb.Optional(ydb.TypeUint64)),
				table.WithPrimaryKeyColumn("serial", "ca"),
			)
		}),
	)
	if err != nil {
		return
	}

	err = table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, path.Join(prefix, "tvm_tags"),
				table.WithColumn("tvm_id", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("service_id", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("resource_id", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("tags", ydb.Optional(ydb.TypeJSON)),
				table.WithColumn("updated_at", ydb.Optional(ydb.TypeUint64)),
				table.WithPrimaryKeyColumn("tvm_id"),
			)
		}),
	)
	if err != nil {
		return
	}

	err = table.Retry(ctx, sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) error {
			return s.CreateTable(ctx, path.Join(prefix, "staff_users"),
				table.WithColumn("uid", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("updated_at", ydb.Optional(ydb.TypeUint64)),
				table.WithColumn("login", ydb.Optional(ydb.TypeString)),
				table.WithPrimaryKeyColumn("uid"),
			)
		}),
	)
	if err != nil {
		return
	}
	return
}

func generateTTL() uint64 {
	return uint64(time.Now().AddDate(0, 0, random(unknownSecretTTLLow, unknownSecretTTLHigh)).Unix())
}

func random(min, max int) int {
	return rand.Intn(max-min) + min
}
