package config

import (
	"context"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/file"

	"a.yandex-team.ru/security/ant-secret/web/internal/staff"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Config struct {
	Addr            string          `config:"addr"`
	YdbAuthToken    string          `config:"ydb_token,required"`
	YdbDatabase     string          `config:"ydb_database,required"`
	YdbEndpoint     string          `config:"ydb_endpoint,required"`
	OAuthToken      string          `config:"oauth_token,required"`
	InternalToken   string          `config:"internal_token,required"`
	BaseURL         string          `config:"base_url"`
	ExtendedACLDeps []string        `config:"extended_acl_deps"`
	ExtendedACLUIDs map[uint64]bool `config:"-"`
	Excludes        []string        `config:"excludes"`
	Debug           bool            `config:"debug"`
}

func LoadConfig(cfgPath string, parseDeps bool) (*Config, error) {
	loader := confita.NewLoader(
		env.NewBackend(),
		file.NewBackend(cfgPath),
	)

	cfg := Config{
		Debug:   false,
		Addr:    ":80",
		BaseURL: "https://ant.sec.yandex-team.ru",
	}
	err := loader.Load(context.Background(), &cfg)
	if err != nil {
		return nil, err
	}

	if parseDeps {
		cfg.ParseDeps()
	}

	return &cfg, err
}

func (c Config) CheckExtendedACL(uid uint64) bool {
	allow, ok := c.ExtendedACLUIDs[uid]
	return ok && allow
}

func (c *Config) ParseDeps() {
	c.ExtendedACLUIDs = make(map[uint64]bool)
	for _, dep := range c.ExtendedACLDeps {
		uids, err := staff.DepartmentUsers(c.OAuthToken, dep)
		if err != nil {
			simplelog.Error("failed to fetch dep uids", "dep", dep, "err", err)
			continue
		}

		for _, uid := range uids {
			c.ExtendedACLUIDs[uid] = true
		}
	}
}
