package server

import (
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/security/ant-secret/web/internal/srvbb"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

func (s Server) internalAuthMiddleware(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		if !s.checkInternalAuth(c) {
			return c.JSON(http.StatusUnauthorized, echo.Map{
				"ok":    false,
				"error": "unauthorized",
			})
		}

		return next(c)
	}
}

func (s Server) checkInternalAuth(c echo.Context) bool {
	token := c.Request().Header.Get("X-Internal-Token")
	return token != "" &&
		s.cfg.InternalToken != "" &&
		token == s.cfg.InternalToken
}

func (s Server) getSessionAuth(c echo.Context) (blackbox.User, error) {
	if c.Request().Header.Get("Authorization") != "" {
		return s.authByOAuth(c)
	}

	return s.authByCookies(c)
}

func (s Server) authByCookies(c echo.Context) (blackbox.User, error) {
	sessid, err := c.Cookie("Session_id")
	if err != nil {
		return blackbox.User{}, err
	}

	rsp, err := srvbb.Intranet.SessionID(
		c.Request().Context(),
		blackbox.SessionIDRequest{
			SessionID: sessid.Value,
			UserIP:    s.getUserIP(c),
			Host:      c.Request().Host,
		})

	if err != nil {
		simplelog.Error("failed to check user session", "err", err)
		return blackbox.User{}, err
	}

	return rsp.User, nil
}

func (s Server) authByOAuth(c echo.Context) (blackbox.User, error) {
	token := c.Request().Header.Get("Authorization")
	if len(token) <= 6 || token[:6] != "OAuth " {
		simplelog.Warn("failed to parse OAuth token")
		return blackbox.User{}, nil
	}

	rsp, err := srvbb.Intranet.OAuth(
		c.Request().Context(),
		blackbox.OAuthRequest{
			OAuthToken: token[6:],
			UserIP:     s.getUserIP(c),
		})

	if err != nil {
		simplelog.Error("failed to check OAuth token", "err", err)
		return blackbox.User{}, err
	}

	return rsp.User, nil
}
