package server

import (
	"crypto/sha1"
	"fmt"
	"net/http"
	"net/url"
	"strings"

	"github.com/labstack/echo/v4"
)

type (
	ValidateRequest struct {
		Secret    string `form:"secret" json:"secret"`
		SkipKnown bool   `form:"skip_known" json:"skip_known"`
	}

	TouchRequest struct {
		Secret string      `json:"secret"`
		Meta   interface{} `json:"meta,omitempty"`
	}
)

func (s Server) getUserIP(c echo.Context) string {
	ip := c.Request().Header.Get("X-Forwarded-For-Y")
	if ip == "" {
		// Use default IP
		ip = "2a02:6b8:0:3400::519"
	}

	return ip
}

func (s Server) makeValidationURL(secret string) string {
	return fmt.Sprintf("%s/#secret=%s", s.cfg.BaseURL, url.QueryEscape(secret))
}

func parseSecret(c echo.Context) (secret string, ok bool) {
	var input ValidateRequest
	if c.Bind(&input) != nil || input.Secret == "" {
		_ = c.JSON(http.StatusBadRequest, echo.Map{
			"ok":    false,
			"error": "no secret",
		})
		return
	}

	ok = true
	secret = strings.TrimSpace(input.Secret)
	return
}

func calcSha1(secret string) string {
	return fmt.Sprintf("%x", sha1.Sum([]byte(secret)))
}
