package server

import (
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net"
	"net/http"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/ant-secret/web/internal/searcher"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var (
	pingUpstreams = map[string]string{
		// TVM
		// TODO(buglloc): fix me to something more generic
		"tvmtool": "http://localhost:2/tvm/ping",
		// Search Server
		"secret-search": fmt.Sprintf("%s://%s/ping", searcher.UpstreamScheme, searcher.UpstreamHost),
	}

	pingClient = &http.Client{
		Transport: &http.Transport{
			Proxy: http.ProxyFromEnvironment,
			DialContext: (&net.Dialer{
				Timeout:   500 * time.Millisecond,
				KeepAlive: 60 * time.Second,
			}).DialContext,
		},
		Timeout: 1 * time.Second,
		CheckRedirect: func(req *http.Request, via []*http.Request) error {
			return http.ErrUseLastResponse
		},
	}
)

func (s *Server) pingHandler(c echo.Context) error {
	ok := true
	for upstream, url := range pingUpstreams {
		res, err := pingClient.Get(url)
		if err != nil {
			simplelog.Error("failed to ping upstream",
				"upstream", upstream,
				"url", url,
				"err", err,
			)
			ok = false
			continue
		}

		_, _ = io.CopyN(ioutil.Discard, res.Body, 128<<10)
		_ = res.Body.Close()

		if res.StatusCode != 200 {
			simplelog.Error("failed to ping upstream",
				"upstream", upstream,
				"status_code", res.StatusCode,
				"err", "not 200 status code",
			)
			ok = false
			continue
		}
	}

	if err := s.db.Ping(); err != nil {
		simplelog.Error("failed to ping DB",
			"err", err,
		)
		ok = false
	}

	if ok {
		return c.String(http.StatusOK, "ok")
	}

	return errors.New("failed to check all upstreams")
}
