package server

import (
	"context"
	"net/http"

	"github.com/karlseguin/ccache/v2"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/security/ant-secret/web/internal/cache"
	"a.yandex-team.ru/security/ant-secret/web/internal/config"
	"a.yandex-team.ru/security/ant-secret/web/internal/unistat"
	"a.yandex-team.ru/security/ant-secret/web/internal/validators"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type (
	Server struct {
		ctx         context.Context
		cfg         *config.Config
		cache       *ccache.Cache
		db          *cache.DB
		statCounter *unistat.Counter
		//statSender  *stat.Sender
	}
)

func NewServer(ctx context.Context, cfg *config.Config) (*Server, error) {
	return &Server{
		ctx: ctx,
		cfg: cfg,
	}, nil
}

func (s *Server) onStart() (err error) {
	s.statCounter = unistat.NewCounter()
	//s.statSender = stat.NewSender()
	//s.statSender.Start()
	s.cache = ccache.New(ccache.Configure().MaxSize(100 * 1024 * 1024))
	s.db, err = cache.NewDB(
		s.ctx,
		cache.Config{
			AuthToken: s.cfg.YdbAuthToken,
			Database:  s.cfg.YdbDatabase,
			Endpoint:  s.cfg.YdbEndpoint,
		},
	)

	if err != nil {
		return
	}
	return
}

func (s *Server) onStop() (err error) {
	//if s.statSender != nil {
	//	s.statSender.Stop()
	//}

	if s.db != nil {
		err = s.db.Close()
		if err != nil {
			return
		}
	}

	s.cache.Stop()
	return
}

func (s *Server) ListenAndServe(addr string) error {
	e := echo.New()
	e.Debug = s.cfg.Debug

	err := s.onStart()
	if err != nil {
		return err
	}

	defer func() {
		if err := s.onStop(); err != nil {
			simplelog.Error("failed to cleanup server", "err", err)
		}
	}()

	e.GET("/ping", s.pingHandler)
	e.File("/favicon.ico", "./runtime/static/favicon.ico")
	e.Static("/static", "./runtime/static")

	e.GET("/", func(c echo.Context) error {
		user, err := s.getSessionAuth(c)
		if err != nil || user.ID == 0 {
			return c.Redirect(
				http.StatusFound,
				"https://passport.yandex-team.ru/passport?mode=auth&retpath="+s.cfg.BaseURL)
		}

		return c.File("./runtime/pages/index.html")
	})

	e.GET("/version", func(c echo.Context) error {
		return c.String(200, buildinfo.Info.ProgramVersion)
	})

	e.GET("/unistat", s.unistatHandler)
	e.POST("/api/v1/solomon/push", s.pushSolomonHandler)

	e.POST("/api/v1/check/hook", s.newSecretSearchHandler("/hook"))
	e.POST("/api/v1/check/diff", s.newSecretSearchHandler("/diff"))
	e.POST("/api/v1/check/plain", s.newSecretSearchHandler("/content"))

	e.POST("/api/v1/validate", s.validateAllHandler)
	e.POST("/api/v1/validate/ssl", s.validateSslHandler)
	for _, v := range validators.Validators {
		e.POST("/api/v1/validate/"+v.RouteName, s.newValidatorHandler(v))
	}

	e.POST("/api/v1/known", s.knownHandler, s.internalAuthMiddleware)
	e.POST("/api/v1/touch-known", s.touchKnownHandler, s.internalAuthMiddleware)

	return e.Start(addr)
}
