package server

import (
	"net/http"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/slices"
	"a.yandex-team.ru/security/ant-secret/web/internal/st"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/ant-secret/web/internal/validators"
	"a.yandex-team.ru/security/ant-secret/web/internal/validators/ssl"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	unknownSecretTTL = 24 * 2 * time.Hour
	knownSecretTTL   = 10 * time.Minute
	failSecretTTL    = 1 * time.Minute
)

type (
	SecretInfo struct {
		Ok             bool                   `json:"ok"`
		Valid          bool                   `json:"valid"`
		YisKnows       bool                   `json:"yis_knows"`
		System         string                 `json:"system,omitempty"`
		Type           string                 `json:"type,omitempty"`
		Users          string                 `json:"users,omitempty"`
		Owners         []string               `json:"owners,omitempty"`
		SecAlerts      []string               `json:"sec_alerts,omitempty"`
		ValidationURL  string                 `json:"validation_url,omitempty"`
		AdditionalInfo map[string]interface{} `json:"additional_info,omitempty"`
		InternalInfo   map[string]interface{} `json:"-"`
	}
)

func (s *Server) validateAllHandler(c echo.Context) error {
	secret, ok := parseSecret(c)
	if !ok {
		return nil
	}

	user, err := s.getSessionAuth(c)
	if err != nil || user.ID == 0 {
		return c.JSON(http.StatusUnauthorized, echo.Map{
			"ok":    false,
			"error": "Unauthorized",
		})
	}

	secretHash := calcSha1(secret)
	if slices.ContainsString(s.cfg.Excludes, secretHash) {
		return c.JSON(http.StatusOK, SecretInfo{
			Ok:    true,
			Valid: false,
		})
	}

	cached := s.cache.Get(secretHash)
	var result *SecretInfo
	if cached != nil && !cached.Expired() && cached.Value() != nil {
		result = cached.Value().(*SecretInfo)
	} else {
		validatorCtx := validator.Context{
			DB:     s.db,
			Secret: secret,
			Sha1:   secretHash,
			UserIP: s.getUserIP(c),
		}

		result, _ = s.searchSecret(validatorCtx)
		var err error
		result.YisKnows, err = s.db.IsKnown(secretHash)
		if err != nil {
			simplelog.Error("failed to check known secret", "err", err)
		}
		s.cache.Set(secretHash, result, time.Minute)
	}

	if sessionID, err := c.Request().Cookie("Session_id"); err == nil && sessionID.Value != "" {
		result.SecAlerts = st.GetSecAlerts(secretHash, sessionID.Value)
	}

	if result.InternalInfo != nil && s.cfg.CheckExtendedACL(user.ID) {
		if result.AdditionalInfo == nil {
			result.AdditionalInfo = result.InternalInfo
		} else {
			for k, v := range result.InternalInfo {
				result.AdditionalInfo[k] = v
			}
		}
	}

	return c.JSON(http.StatusOK, result)
}

func (s Server) searchSecret(ctx validator.Context) (secretInfo *SecretInfo, exists bool) {
	for _, v := range validators.Validators {
		if !v.Match(ctx) {
			continue
		}

		if info, valid, ok := v.Validate(ctx); ok && valid {
			exists = true
			secretInfo = &SecretInfo{
				Ok:             true,
				Valid:          true,
				System:         v.Name,
				Type:           info.Type,
				Users:          info.User,
				Owners:         info.Owners,
				AdditionalInfo: info.AdditionalInfo,
				InternalInfo:   info.InternalInfo,
				ValidationURL:  s.makeValidationURL(ctx.Secret),
			}
			return
		}
	}

	secretInfo = &SecretInfo{
		Ok: false,
	}
	return
}

func (s Server) validateSslHandler(c echo.Context) error {
	var request ssl.ValidationRequest
	err := c.Bind(&request)
	if err != nil {
		return c.JSON(http.StatusBadRequest, echo.Map{
			"ok":    false,
			"error": err,
		})
	}

	valid, revoked := ssl.Validate(s.db, request)
	return c.JSON(http.StatusOK, echo.Map{
		"ok":      true,
		"valid":   valid,
		"revoked": revoked,
	})
}

func (s *Server) newValidatorHandler(v validator.Validator) echo.HandlerFunc {
	return func(c echo.Context) error {
		var input ValidateRequest
		if c.Bind(&input) != nil || input.Secret == "" {
			return c.JSON(http.StatusBadRequest, echo.Map{
				"ok":    false,
				"error": "no secret",
			})
		}

		secret := strings.TrimSpace(input.Secret)
		sha1sum := calcSha1(secret)
		if slices.ContainsString(s.cfg.Excludes, sha1sum) {
			return c.JSON(http.StatusOK, SecretInfo{
				Ok:    true,
				Valid: false,
			})
		}

		cacheKey := v.RouteName + sha1sum
		if input.SkipKnown {
			cacheKey += "_wok"
		}

		cached := s.cache.Get(cacheKey)
		var result *SecretInfo
		switch {
		case cached != nil && !cached.Expired() && cached.Value() != nil:
			// Get from hot cache
			s.statCounter.AddHotHit()
			result = cached.Value().(*SecretInfo)
		case s.isUnknownSecret(sha1sum, v.RouteName, v.Cacheable, input.SkipKnown):
			// This is not a secret
			s.statCounter.AddUnknownCacheHit()
			result = &SecretInfo{
				Ok:    true,
				Valid: false,
			}
			s.cache.Set(cacheKey, result, unknownSecretTTL)
		default:
			// check it
			validatorCtx := validator.Context{
				DB:     s.db,
				Secret: secret,
				Sha1:   sha1sum,
				UserIP: s.getUserIP(c),
			}
			secretInfo, valid, ok := v.Validate(validatorCtx)
			var ttl time.Duration
			switch {
			case !ok:
				s.statCounter.AddFailedFetch()
				ttl = failSecretTTL
			case ok && !valid:
				s.statCounter.AddUnknownFetch()
				ttl = unknownSecretTTL
				// cache invalid secret
				if v.Cacheable {
					err := s.db.TouchUnknown(sha1sum, v.RouteName)
					if err != nil {
						simplelog.Error("failed to touch secret", "err", err)
					}
				}
			case ok && valid:
				s.statCounter.AddKnownFetch()
				ttl = knownSecretTTL
			}

			result = &SecretInfo{
				Ok:    ok,
				Valid: valid,
			}

			if ok && valid {
				result.System = v.Name
				result.Type = secretInfo.Type
				result.Users = secretInfo.User
				result.Owners = secretInfo.Owners
				result.ValidationURL = s.makeValidationURL(secret)
				result.AdditionalInfo = secretInfo.AdditionalInfo
				result.InternalInfo = secretInfo.InternalInfo
			}

			s.cache.Set(cacheKey, result, ttl)
		}

		if result.InternalInfo != nil /*&& isInternalRequest(c)*/ {
			if result.AdditionalInfo == nil {
				result.AdditionalInfo = result.InternalInfo
			} else {
				for k, v := range result.InternalInfo {
					result.AdditionalInfo[k] = v
				}
			}
		}

		return c.JSON(http.StatusOK, result)
	}
}

func (s Server) isUnknownSecret(sha1, validator string, cacheable bool, skipKnown bool) bool {
	var unknown bool
	var err error

	switch {
	case skipKnown && cacheable:
		unknown, err = s.db.IsKnownOrUnknown(sha1, validator)
	case skipKnown:
		unknown, err = s.db.IsKnown(sha1)
	case cacheable:
		unknown, err = s.db.IsUnknown(sha1, validator)
	default:
		return false
	}

	if err != nil {
		simplelog.Error("failed to check unknown secret", "err", err)
	}
	return unknown
}
