package stat

import (
	"bytes"
	"net/http"
	"sync"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/ant-secret/web/internal/httpclient"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

const (
	InsertTimeout = 20 * time.Millisecond
	UpstreamURL   = "http://api.solomon.search.yandex.net/push/json"
	WorkersCount  = 1
)

type (
	message []byte

	Sender struct {
		channel      chan message
		workersCount int
		wg           *sync.WaitGroup
	}
)

func NewSender() *Sender {
	return &Sender{
		channel:      make(chan message, 1000),
		workersCount: WorkersCount,
		wg:           &sync.WaitGroup{},
	}
}

func (s *Sender) Start() {
	for i := 0; i < s.workersCount; i++ {
		s.wg.Add(1)

		go func() {
			defer s.wg.Done()

			for msg := range s.channel {
				if msg == nil {
					continue
				}

				req, err := http.NewRequest("POST", UpstreamURL, bytes.NewBuffer(msg))
				if err != nil {
					simplelog.Warn("failed to make stat request", "err", err)
					continue
				}
				req.Header.Set(echo.HeaderContentType, echo.MIMEApplicationJSON)
				res, err := httpclient.Client.Do(req)
				if err != nil {
					simplelog.Error("cannot send stat into upstream", "err", err)
					continue
				}

				yahttp.GracefulClose(res.Body)
			}
		}()
	}
}

func (s *Sender) Stop() {
	close(s.channel)
	s.wg.Wait()
}

func (s *Sender) PushMsg(msg message) {
	select {
	case s.channel <- msg:
	case <-time.After(InsertTimeout):
		simplelog.Warn("stat queue is full: dropping request not putting it on queue")
	}
}
