package stringutils

import (
	"bytes"
	"math"
	"strings"
)

func ShannonEntropy(data string, characters []rune) float64 {
	dataLen := len(data)
	if dataLen == 0 {
		return 0
	}

	entropy := 0.0
	for _, c := range characters {
		px := float64(strings.Count(data, string(c))) / float64(dataLen)
		if px > 0 {
			entropy += -px * math.Log2(px)
		}

	}
	return entropy
}

func HaveToken(line string, tokens []string) bool {
	if len(tokens) == 0 {
		return true
	}

	for _, token := range tokens {
		if strings.Contains(line, token) {
			return true
		}
	}

	return false
}

func RuneContains(slice []rune, search rune) bool {
	for _, value := range slice {
		if value == search {
			return true
		}
	}
	return false
}

func StringToList(str *string) []string {
	var result = make([]string, 0)
	for _, val := range strings.Split(*str, ",") {
		result = append(result, strings.Trim(val, " "))
	}

	return result
}

func SplitByChars(str string, characters []rune, threshold int) []string {
	var result = make([]string, 0)
	var current bytes.Buffer
	count := 0
	for _, c := range str {
		if RuneContains(characters, c) {
			count++
			current.WriteRune(c)
		} else {
			if count > threshold {
				result = append(result, current.String())
			}
			count = 0
			current.Reset()
		}
	}

	if count > threshold {
		result = append(result, current.String())
	}
	return result
}

func IsBase64(str string) (base64 bool) {
	if len(str)%4 != 0 {
		// Base64 string must be multiple of 4
		return
	}

	var haveUpper bool
	var haveLower bool
	var haveDigit bool
	for _, c := range str {
		switch {
		case '0' <= c && c <= '9':
			haveDigit = true
		case 'a' <= c && c <= 'z':
			haveLower = true
		case 'A' <= c && c <= 'Z':
			haveUpper = true
		case c == '+' || c == '/' || c == '=':
			// pass
		default:
			return
		}
	}

	// by statistic base64 probably contain lower/upper char and digit
	return haveUpper && haveLower && haveDigit
}

func IsBase64Url(str string) (base64 bool) {
	var haveUpper bool
	var haveLower bool
	var haveDigit bool
	for _, c := range str {
		switch {
		case '0' <= c && c <= '9':
			haveDigit = true
		case 'a' <= c && c <= 'z':
			haveLower = true
		case 'A' <= c && c <= 'Z':
			haveUpper = true
		case c == '*' || c == '-' || c == '_':
			// pass
		default:
			return
		}
	}

	// by statistic base64 probably contain lower/upper char and digit
	return haveUpper && haveLower && haveDigit
}
