package github

import (
	"encoding/json"
	"net/http"
	"regexp"
	"strings"

	"a.yandex-team.ru/security/ant-secret/web/internal/httpclient"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	ghInternalURL = "https://github.yandex-team.ru/api/v3/user"
	//ghExternalURL = "https://api.github.com/user"
)

type (
	ghInfo struct {
		Login  string `json:"login"`
		Scopes string `json:"scopes"`
	}
)

var (
	tokenRe = regexp.MustCompile(`^[a-f0-9]{40}$`)
)

func Match(ctx validator.Context) (matched bool) {
	return tokenRe.MatchString(ctx.Secret)
}

func Validate(ctx validator.Context) (info *validator.Info, valid bool, ok bool) {
	var gh *ghInfo
	gh, valid, ok = checkToken(ghInternalURL, ctx.Secret)

	if !ok {
		return
	}

	if valid {
		info = &validator.Info{
			Type:   "GithubInternal",
			User:   gh.Login,
			Owners: validator.StaffOwners(gh.Login),
			AdditionalInfo: map[string]interface{}{
				"scopes": gh.Scopes,
			},
		}
	}

	// Disable external GH to prevent tokens leakage
	//if res, valid := checkToken(ghExternalURL, token); valid {
	//	ok = true
	//	info = &validator.Info{
	//		Type: "GithubExternal",
	//		User: res.Login,
	//		AdditionalInfo: map[string]interface{}{
	//			"scopes": res.Scopes,
	//		},
	//	}
	//	return
	//}
	return
}

func checkToken(apiURL string, token string) (info *ghInfo, valid bool, ok bool) {
	req, err := http.NewRequest("GET", apiURL, nil)
	if err != nil {
		simplelog.Error("failed to make GH request", "err", err)
		return
	}
	// Force v3 api version: https://developer.github.com/v3/media/
	req.Header.Set("Accept", "application/vnd.github.v3+json")
	req.Header.Set("Authorization", "token "+token)

	res, err := httpclient.Client.Do(req)
	if err != nil {
		simplelog.Error("failed to validate GH Token", "uri", apiURL, "err", err)
		return
	}
	defer httpclient.GracefulClose(res.Body)

	switch res.StatusCode {
	case 401, 403:
		// invalid token, it's fine
		ok = true
		return
	case 200:
		// pass
		break
	default:
		simplelog.Error("failed to fetch GH response", "status_code", res.StatusCode)
		return
	}

	var result ghInfo
	err = json.NewDecoder(res.Body).Decode(&result)
	if err != nil {
		simplelog.Error("failed to parse GH response", "err", err)
		return
	}

	if rScopes := res.Header.Get("X-OAuth-Scopes"); rScopes != "" {
		result.Scopes = strings.Replace(rScopes, ", ", ",", -1)
	} else {
		// https://developer.github.com/enterprise/2.14/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/
		// (no scope)	Grants read-only access to public information (includes public user profile info, public repository info, and gists)
		result.Scopes = "read-public"
	}

	ok = true
	valid = true
	info = &result
	return
}
