package slack

import (
	"encoding/json"
	"net/url"
	"regexp"

	"a.yandex-team.ru/security/ant-secret/web/internal/httpclient"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	slackURL = "https://slack.com/api/auth.test"
)

type (
	userInfo struct {
		User string `json:"user"`
		Team string `json:"team"`
	}

	slackResponse struct {
		userInfo
		Ok bool `json:"ok"`
	}
)

var (
	// https://api.slack.com/docs/token-types
	slackRe = regexp.MustCompile(`^xox[baprs]-(?:\d{5,}-)+\w+$`)
)

func Match(ctx validator.Context) (matched bool) {
	return slackRe.MatchString(ctx.Secret)
}

func Validate(ctx validator.Context) (info *validator.Info, valid bool, ok bool) {
	var slackInfo userInfo
	slackInfo, valid, ok = checkToken(ctx.Secret)
	if valid {
		info = &validator.Info{
			Type: "Slack",
			AdditionalInfo: map[string]interface{}{
				"user": slackInfo.User,
				"team": slackInfo.Team,
			},
		}
	}
	return
}

func checkToken(token string) (info userInfo, valid bool, ok bool) {
	res, err := httpclient.Client.PostForm(slackURL, url.Values{
		"token": {token},
	})
	if err != nil {
		simplelog.Error("failed to validate Slack Token", "err", err)
		return
	}
	defer httpclient.GracefulClose(res.Body)

	ok = true

	if res.StatusCode != 200 {
		simplelog.Error("failed to fetch Slack response", "status_code", res.StatusCode)
		return
	}

	var slackRes slackResponse
	err = json.NewDecoder(res.Body).Decode(&slackRes)
	if err != nil {
		simplelog.Error("failed to parse Slack response", "err", err)
		return
	}

	if slackRes.Ok {
		info = slackRes.userInfo
		valid = true
	}
	return
}
