package telegram

import (
	"encoding/json"
	"fmt"
	"net/url"
	"regexp"

	"a.yandex-team.ru/security/ant-secret/web/internal/httpclient"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	tgURL = "https://api.telegram.org/bot%s/getMe"
)

type (
	userInfo struct {
		FirstName string `json:"first_name"`
		UserName  string `json:"username"`
	}

	tgResponse struct {
		Ok     bool     `json:"ok"`
		Result userInfo `json:"result"`
	}
)

var (
	tokenRe = regexp.MustCompile(`^[0-9]{8,}:[\w-]{35}$`)
)

func Match(ctx validator.Context) (matched bool) {
	return tokenRe.MatchString(ctx.Secret)
}

func Validate(ctx validator.Context) (info *validator.Info, valid bool, ok bool) {
	var tgInfo userInfo
	tgInfo, valid, ok = checkURL(genTokenURL(ctx.Secret))
	if valid {
		info = &validator.Info{
			Type: "Telegram",
			AdditionalInfo: map[string]interface{}{
				"user":       tgInfo.UserName,
				"first_name": tgInfo.FirstName,
			},
		}
	}
	return
}

func checkURL(url string) (info userInfo, valid bool, ok bool) {
	res, err := httpclient.Client.Get(url)
	if err != nil {
		simplelog.Warn("failed to make Telegram request", "err", err)
		return
	}
	defer httpclient.GracefulClose(res.Body)

	switch res.StatusCode {
	case 401, 403:
		// invalid token, it's fine
		ok = true
		return
	case 200:
		ok = true
		// pass
	default:
		simplelog.Error("failed to fetch Telegram response", "status_code", res.StatusCode)
		return
	}

	var tgRes tgResponse
	err = json.NewDecoder(res.Body).Decode(&tgRes)
	if err != nil {
		simplelog.Error("failed to parse Telegram response", "err", err)
		return
	}

	if tgRes.Ok {
		valid = true
		info = tgRes.Result
	}
	return
}

func genTokenURL(token string) string {
	return fmt.Sprintf(tgURL, url.QueryEscape(token))
}
