package tvm

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"regexp"
	"strings"

	"a.yandex-team.ru/security/ant-secret/web/internal/cache"
	"a.yandex-team.ru/security/ant-secret/web/internal/httpclient"
	"a.yandex-team.ru/security/ant-secret/web/internal/srvbb"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	tvmCheckURL = "https://tvm-api.yandex.net/2/check_secret"
	tvmInfoURL  = "https://tvm.yandex-team.ru/client/%d/info"
)

type (
	secretInfo []struct {
		ClientID     uint64   `json:"client_id"`
		ClientName   string   `json:"client_name"`
		EnvType      string   `json:"env_type"`
		Creator      string   `json:"creator"`
		AbcID        uint64   `json:"abc_id"`
		ResourceID   uint64   `json:"resource_id"`
		ResourceTags []string `json:"resource_tags"`
	}

	clientInfo struct {
		Status     string `json:"status"`
		ClientName string `json:"name"`
		EnvType    string `json:"env_type"`
		CreatorUID uint64 `json:"creator_uid"`
	}

	resolvedSecret struct {
		Valid bool
		Info  secretInfo
	}
)

var (
	secretRe       = regexp.MustCompile(`^[A-Za-z0-9_\-]{22}$`)
	productionTags = []uint64{
		2,  // Production
		29, // Prestable
		30, // Unstable
	}
)

func Match(ctx validator.Context) (matched bool) {
	return secretRe.MatchString(ctx.Secret)
}

/*
Logic are pretty simple:
	- Send secret to https://tvm-api.yandex.net/2/check_secret to validate TVM secret
    - If valid - get TVM application info (creator uid and so on_
    - Resolve creator uid thought tvm
*/
func Validate(ctx validator.Context) (info *validator.Info, valid bool, ok bool) {
	tvmTicket, err := srvbb.TVM.GetServiceTicketForAlias(context.Background(), "tvm")
	if err != nil {
		simplelog.Error("failed to get TVM ticket", "err", err)
		return
	}

	var tvmInfo resolvedSecret
	tvmInfo, ok = checkTvmSecret(ctx.Secret, tvmTicket)
	if tvmInfo.Valid {
		info = &validator.Info{
			Type: "TVM",
		}

		var appOk bool
		tvmInfo.Info, appOk, err = extendTvmInfo(ctx, tvmInfo.Info)
		if err != nil {
			ok = false
			return
		}

		if appOk {
			// valid TVM secret MUST have existed TVM Application
			valid = true

			clientIDs := make(map[uint64]bool)
			clientNames := make(map[string]bool)
			creators := make(map[string]bool)
			envTypes := make(map[string]bool)
			abcIDs := make(map[uint64]bool)
			resourceIDs := make(map[uint64]bool)
			resourceTags := make(map[string]bool)
			for _, i := range tvmInfo.Info {
				clientIDs[i.ClientID] = true
				clientNames[i.ClientName] = true
				creators[i.Creator] = true
				abcIDs[i.AbcID] = true
				envTypes[i.EnvType] = true
				resourceIDs[i.ResourceID] = true
				for _, t := range i.ResourceTags {
					resourceTags[t] = true
				}
			}

			for abcID := range abcIDs {
				info.Owners = append(info.Owners, validator.ABCOwner(fmt.Sprint(abcID)))
			}

			for creator := range creators {
				info.Owners = append(info.Owners, validator.StaffOwner(creator))
			}

			info.InternalInfo = map[string]interface{}{
				"client_ids":    joinUIntMapKeys(clientIDs),
				"client_names":  joinStringMapKeys(clientNames),
				"abc_ids":       joinUIntMapKeys(abcIDs),
				"creators":      joinStringMapKeys(creators),
				"env_types":     joinStringMapKeys(envTypes),
				"resource_ids":  joinUIntMapKeys(resourceIDs),
				"resource_tags": joinStringMapKeys(resourceTags),
			}
		}
	}
	return
}

func checkTvmSecret(secret, ticket string) (info resolvedSecret, ok bool) {
	req, err := http.NewRequest("GET", tvmCheckURL, nil)
	if err != nil {
		simplelog.Error("failed make TVM request", "err", err)
		return
	}

	req.Header.Set("X-Ya-Secret", secret)
	req.Header.Set("X-Ya-Service-Ticket", ticket)

	res, err := httpclient.Client.Do(req)
	if err != nil {
		simplelog.Error("failed to check TVM secret", "err", err)
		return
	}
	defer httpclient.GracefulClose(res.Body)

	switch res.StatusCode {
	case 200:
		// This is our secret!
		ok = true
		err = json.NewDecoder(res.Body).Decode(&info.Info)
		if err != nil {
			simplelog.Error("failed to parse TVM response", "err", err)
			return
		}

		info.Valid = info.Info != nil
	case 404:
		// Not a secret
		ok = true
	default:
		// something go wrong :(
		simplelog.Error("unexpected TVM response", "status_code", res.StatusCode)
	}
	return
}

func extendTvmInfo(ctx validator.Context, info secretInfo) (result secretInfo, ok bool, err error) {
	result = make(secretInfo, len(info))
	for i, inf := range info {
		clientInfo, clientOk, cErr := getTvmClientInfo(inf.ClientID)
		if cErr != nil {
			err = cErr
			return
		}

		if clientOk {
			tagsInfo, clientOk, cErr := getTvmTags(ctx, inf.ClientID)
			if cErr != nil {
				err = cErr
				return
			}

			if clientOk {
				ok = true
				inf.ResourceID = tagsInfo.ResourceID
				inf.ResourceTags = make([]string, len(tagsInfo.Tags))
				for i, tag := range tagsInfo.Tags {
					inf.ResourceTags[i] = fmt.Sprintf("[%d]%s", tag.ID, tag.Name)
				}
				inf.ClientName = clientInfo.ClientName
				inf.EnvType = clientInfo.EnvType
				if login, _ := ctx.DB.UIDToLogin(clientInfo.CreatorUID); login != "" {
					inf.Creator = login
				}
			}
		}
		result[i] = inf
	}
	return
}

func getTvmClientInfo(clientID uint64) (info *clientInfo, ok bool, err error) {
	res, hErr := httpclient.Client.Get(fmt.Sprintf(tvmInfoURL, clientID))
	if hErr != nil {
		err = hErr
		return
	}
	defer httpclient.GracefulClose(res.Body)

	switch res.StatusCode {
	case 200:
		var parsed clientInfo
		err = json.NewDecoder(res.Body).Decode(&parsed)
		if err != nil {
			simplelog.Error("failed to parse TVM client info", "client_id", clientID, "err", err)
			return
		}

		if parsed.Status != "ok" {
			err = fmt.Errorf("non OK status: %s", parsed.Status)
			return
		}

		info = &parsed
		ok = true
	case 404:
		break
	default:
		simplelog.Error("failed to get TVM client info", "client_id", clientID, "err", "not 200 status code")
		err = errors.New("not 200 OK")
	}
	return
}

func getTvmTags(ctx validator.Context, clientID uint64) (result cache.TvmInfo, ok bool, err error) {
	result, err = ctx.DB.TvmInfo(clientID)

	if err != nil {
		return
	}

	// Report as a valid TVM service in two cases:
	//   1. We didn't have any tags at all
	//   2. TVM resource have production-like tag
	if len(result.Tags) == 0 {
		ok = true
		return
	}

tags:
	for _, tag := range result.Tags {
		for _, id := range productionTags {
			if tag.ID == id {
				ok = true
				break tags
			}
		}
	}
	return
}

func joinUIntMapKeys(m map[uint64]bool) string {
	var result string
	for k := range m {
		if k == 0 {
			continue
		}

		result += fmt.Sprintf(",%d", k)
	}

	if result == "" {
		return ""
	}

	return result[1:]
}

func joinStringMapKeys(m map[string]bool) string {
	var result string
	for k := range m {
		k = strings.TrimSpace(k)
		if k == "" {
			continue
		}

		result += "," + k
	}

	if result == "" {
		return ""
	}

	return result[1:]
}
