package xiva

import (
	"encoding/json"
	"net/http"
	"regexp"

	"a.yandex-team.ru/security/ant-secret/web/internal/httpclient"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	validationURL = "http://push.yandex-team.ru:1080/check_token"
)

type (
	environments map[string]struct {
		Type        string `json:"type"`
		Description string `json:"description"`
		Service     string `json:"service"`
		Owner       string `json:"owner"`
	}

	xivaResponse struct {
		Found        bool         `json:"token_found"`
		Environments environments `json:"environments"`
	}
)

var (
	tokenRe = regexp.MustCompile(`^[a-f0-9]{40}$`)
)

func Match(ctx validator.Context) (matched bool) {
	return tokenRe.MatchString(ctx.Secret)
}

func Validate(ctx validator.Context) (info *validator.Info, valid bool, ok bool) {
	req, err := http.NewRequest("GET", validationURL, nil)
	if err != nil {
		simplelog.Error("failed make Xiva request", "err", err)
		return
	}

	req.Header.Set("X-XivaToken", ctx.Secret)

	res, err := httpclient.Client.Do(req)
	if err != nil {
		simplelog.Error("failed to make Xiva request", "err", err)
		return
	}
	defer httpclient.GracefulClose(res.Body)

	if res.StatusCode != 200 {
		simplelog.Error("failed to fetch Xiva response", "status_code", res.StatusCode)
		return
	}

	var response xivaResponse
	err = json.NewDecoder(res.Body).Decode(&response)
	if err != nil {
		simplelog.Error("failed to parse Xiva response", "err", err)
		return
	}

	ok = true
	if response.Found {
		valid = true
		info = &validator.Info{
			Type: "Xiva",
			AdditionalInfo: map[string]interface{}{
				"environments": response.Environments,
			},
		}
	}
	return
}
