package yandexoauth

import (
	"context"
	"fmt"
	"regexp"
	"strings"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/security/ant-secret/web/internal/srvbb"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var (
	oAuthRe *regexp.Regexp
)

func init() {
	/*
		We support 3 types of oauth tokens:
		  1. old legacy token - md5 hash, e.g.:
			03c7c0ace395d80182db07ae2c30f034
		  2. token with embedded info - base64url(uid, clid, shard, random), [a-zA-Z0-9_-]+, e.g.:
			AQAD-qJSJpcjAAADwHcUZ2Serk5EmBjzGpemfJQ
		  3. stateless token - ver.uid.client_id.expires.token_id.key_id.iv.data.tag, e.g.:
			1.0.123.2222222222.1111222233000.12350.YRf1SNShCFuRY56E.KO0b4NfG.k0sB3ImkZb5MarhDjgyU5Q
	*/

	legacy := `[a-f0-9]{32}`
	// TODO(buglloc): remove prefix?
	embeddedV2 := `A[\w\-]{38}`
	embeddedV3 := `y[0-9]_[A-Za-z0-9_\-]{55}`
	stateless := `\d+\.\d+\.\d+\.\d+\.\d+\.\d+\.[\w\-]+\.[\w\-]+\.[\w\-]+`
	oAuthRe = regexp.MustCompile(fmt.Sprintf(`^(?:%s)$`,
		strings.Join([]string{legacy, embeddedV2, embeddedV3, stateless}, "|")))
}

func Match(ctx validator.Context) (matched bool) {
	return oAuthRe.MatchString(ctx.Secret)
}

func Validate(ctx validator.Context) (info *validator.Info, valid bool, ok bool) {
	var bbRsp *blackbox.OAuthResponse

	// Internal
	bbRsp, valid, ok = checkCtx(ctx, srvbb.Intranet)
	if !ok {
		return
	}

	if valid {
		info = &validator.Info{Type: "YOAuthInternal"}
		extendValidatorInfo(info, bbRsp)
		return
	}

	// External
	bbRsp, valid, ok = checkCtx(ctx, srvbb.Production)
	if !ok {
		return
	}

	if valid {
		info = &validator.Info{Type: "YOAuthExternal"}
		extendValidatorInfo(info, bbRsp)
		return
	}
	return
}

func checkCtx(ctx validator.Context, bb blackbox.Client) (rsp *blackbox.OAuthResponse, valid bool, ok bool) {
	bbRsp, err := bb.OAuth(
		context.Background(),
		blackbox.OAuthRequest{
			OAuthToken: ctx.Secret,
			UserIP:     ctx.UserIP,
		},
	)

	if err != nil {
		switch err.(type) {
		case *blackbox.UnauthorizedError, *blackbox.StatusError:
			// that's fine
			ok = true
		default:
			simplelog.Error("failed to check oauth token", "err", err)
		}
		return
	}

	ok = true
	valid = true
	rsp = bbRsp
	return
}

func extendValidatorInfo(info *validator.Info, bbRsp *blackbox.OAuthResponse) {
	info.User = bbRsp.User.Login
	info.Owners = validator.StaffOwners(bbRsp.User.Login)
	info.AdditionalInfo = map[string]interface{}{
		"client_id":   bbRsp.ClientID,
		"token_id":    bbRsp.TokenID,
		"client_name": bbRsp.ClientName,
		"device_id":   bbRsp.DeviceID,
		"device_name": bbRsp.DeviceName,
		"scopes":      strings.Join(bbRsp.Scopes, ","),
		"create_time": bbRsp.CreateTime,
		"issue_time":  bbRsp.IssueTime,
	}
}
