package yandexsession

import (
	"context"
	"regexp"
	"strings"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/security/ant-secret/web/internal/srvbb"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

var (
	/*
			https://wiki.yandex-team.ru/passport/sessionid/#formatv3spodderzhkojjmultiavtorizacijj
			Examples:
				- 3:1552306280.5.0.1552217798723:Z8FsXw:1d.1|126480966.88482.2.2:88482|196108.640870.XXXXXXXXXXXXXXXXXXXX-6SznOg
		        - 3:1552308886.5.0.1527855241846:UhFYTQ:88.1|1120000000001700.-1.400|1120000000026268.-1.0|1120000000102258.-1.0|1120000000081886.-1.400|1120000000038691.-1.0|1120000000123201.-1.0|134332.159643.XXXXXXXXXXXXXXXXXXXX-6SznOg
				- 3:1552308886.5.5.1527855241846:UhFYTQ:88.1|1120000000001700.20469790.2402.2:20469790|1120000000026268.9690762.2002.2:9690762|1120000000102258.20726003.2002.0:3.2:20726003|1120000000081886.21172253.2402.2:21172253|1120000000038691.24012992.2002.2:24012992|1120000000123201.24453645.2002.2:24453645|134332.183007.XXXXXXXXXXXXXXXXXXXX-6SznOg
				- 3:1551182683.5.0.1540542645970:LDh8YjAwYU0MBAAAuAYCKg:43.1|88994589.0.302|195482.741222.XXXXXXXXXXXXXXXXXXXX-6SznOg
	*/

	sessionCookieRe = regexp.MustCompile(`^(?:3:\d{10}\.\d+\.[0-9a-f]{1,2}\.(?:\d+:[\w\-]+:[\w\-]+|\d+)(?:\.[\w:\-]+)*\|(?:\d+\.[\d\-]+\.\d+(?:\.[\w:\-]+)*\|)+[\w:\-]+\.\d+\.[\w\-]{27})$`)
)

func Match(ctx validator.Context) (matched bool) {
	return sessionCookieRe.MatchString(ctx.Secret)
}

func Validate(ctx validator.Context) (info *validator.Info, valid bool, ok bool) {
	var bbRsp *blackbox.MultiSessionIDResponse

	// Internal
	bbRsp, valid, ok = checkCtx(ctx, "ant.yandex-team.ru", srvbb.Intranet)
	if !ok {
		return
	}

	if valid {
		info = &validator.Info{Type: "YSessionInternal"}
		extendValidatorInfo(info, bbRsp)
		return
	}

	// External
	bbRsp, valid, ok = checkCtx(ctx, "ant.yandex.ru", srvbb.Production)
	if !ok {
		return
	}

	if valid {
		info = &validator.Info{Type: "YSessionExternal"}
		extendValidatorInfo(info, bbRsp)
		return
	}
	return
}

func checkCtx(ctx validator.Context, host string, bb blackbox.Client) (rsp *blackbox.MultiSessionIDResponse, valid bool, ok bool) {
	bbResponse, err := bb.MultiSessionID(
		context.Background(),
		blackbox.MultiSessionIDRequest{
			SessionID: ctx.Secret,
			UserIP:    ctx.UserIP,
			Host:      host,
		})

	if err != nil {
		switch err.(type) {
		case *blackbox.UnauthorizedError, *blackbox.StatusError:
			// that's fine
			ok = true
		default:
			simplelog.Error("failed to check session id", "err", err)
		}
		return
	}

	ok = true
	// session_id cookie may be valid, but don't have any valid users %)
	valid = len(bbResponse.Users) > 0
	rsp = bbResponse
	return
}

func extendValidatorInfo(info *validator.Info, bbRsp *blackbox.MultiSessionIDResponse) {
	users := make([]string, len(bbRsp.Users))
	for i, u := range bbRsp.Users {
		users[i] = u.Login
	}

	info.User = strings.Join(users, ",")
	info.Owners = validator.StaffOwners(users...)
}
