package ycauth

import (
	"context"
	"fmt"
	"sync"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials"
	"google.golang.org/grpc/keepalive"

	"a.yandex-team.ru/cloud/iam/accessservice/client/iam-access-service-client-go/v1"
	"a.yandex-team.ru/library/go/certifi"
)

var (
	clients   = make(map[string]cloudauth.AccessServiceClient)
	clientsMu sync.Mutex
)

var (
	creds     credentials.TransportCredentials
	credsOnce sync.Once
)

func getClient(endpoint string) (cloudauth.AccessServiceClient, error) {
	clientsMu.Lock()
	if c, ok := clients[endpoint]; ok {
		clientsMu.Unlock()
		return c, nil
	}
	clientsMu.Unlock()

	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	conn, err := grpc.DialContext(ctx, endpoint,
		grpc.WithTransportCredentials(grpcCreds()),
		grpc.WithUserAgent("AntSecret"),
		grpc.WithKeepaliveParams(keepalive.ClientParameters{
			Time:                10 * time.Second,
			Timeout:             1 * time.Second,
			PermitWithoutStream: true,
		}),
	)
	if err != nil {
		return nil, err
	}

	client := cloudauth.NewAccessServiceClient(conn)
	clientsMu.Lock()
	clients[endpoint] = client
	clientsMu.Unlock()

	return client, nil
}

func grpcCreds() credentials.TransportCredentials {
	credsOnce.Do(func() {
		certPool, err := certifi.NewCertPoolInternal()
		if err != nil {
			panic(fmt.Sprintf("failed to create cert pool: %s", err))
		}

		creds = credentials.NewClientTLSFromCert(certPool, "")
	})

	return creds
}
