package ycauth

import (
	"context"
	"reflect"
	"strings"
	"time"

	grpccodes "google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/cloud/iam/accessservice/client/iam-access-service-client-go/v1"
	"a.yandex-team.ru/security/ant-secret/web/internal/validator"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const timeout = 2 * time.Second

// https://docs.yandex-team.ru/iam-cookbook/6.appendices/endpoints#access-service
var environments = map[string][]string{
	"prod":     {"as.private-api.cloud.yandex.net:4286"},
	"pre-prod": {"as.private-api.cloud-preprod.yandex.net:4286"},
	// need puncher rules
	//"gpn": {
	//	"as.private-api.gpn.yandexcloud.net:14286",
	//	"as.private-api.ycp.gpn.yandexcloud.net:443",
	//},
}

type Subject struct {
	cloudSubject cloudauth.Subject
	env          string
}

func Authenticate(ctx context.Context, cred cloudauth.Authenticatable) (*Subject, error) {
	var lastErr error
	for env, endpoints := range environments {
		for _, endpoint := range endpoints {
			client, err := getClient(endpoint)
			if err != nil {
				lastErr = err
				simplelog.Error("unable to create YC client", "endpoint", endpoint, "err", err.Error())
				continue
			}

			callCtx, cancel := context.WithTimeout(ctx, timeout)
			subject, err := client.Authenticate(callCtx, cred)
			cancel()

			if err != nil {
				// that's fine, just invalid or expired credential
				if st, ok := status.FromError(err); ok && st.Code() == grpccodes.Unauthenticated {
					continue
				}

				lastErr = err
				simplelog.Error("unable to check YC credential", "endpoint", endpoint, "err", err.Error())
				continue
			}

			return &Subject{
				cloudSubject: subject,
				env:          env,
			}, nil
		}

	}

	return nil, lastErr
}

func PutSubjectToValidInfo(subject *Subject, dst *validator.Info) *validator.Info {
	if subject.cloudSubject == nil {
		return dst
	}

	if dst.AdditionalInfo == nil {
		dst.AdditionalInfo = make(map[string]interface{})
	}

	switch s := subject.cloudSubject.(type) {
	case cloudauth.AnonymousAccount:
		// ¯\_(⊙︿⊙)_/¯
	case cloudauth.UserAccount:
		dst.AdditionalInfo["account_id"] = s.ID
	case cloudauth.ServiceAccount:
		dst.AdditionalInfo["account_id"] = s.ID
		dst.AdditionalInfo["folder_id"] = s.FolderID
	}

	subjectKind := reflect.TypeOf(subject.cloudSubject).String()
	if idx := strings.IndexByte(subjectKind, '.'); idx > 0 && idx != len(subjectKind) {
		subjectKind = subjectKind[idx+1:]
	}

	dst.AdditionalInfo["kind"] = subjectKind
	dst.AdditionalInfo["env"] = subject.env
	return dst
}
