#include <ydb/library/yql/public/udf/udf_value.h>
#include <ydb/library/yql/public/udf/udf_helpers.h>
#include <ydb/library/yql/public/udf/udf_type_builder.h>

#include <security/ant-secret/secret-search/public/cpp/source.h>
#include <security/ant-secret/secret-search/public/cpp/searcher.h>
#include <security/ant-secret/secret-search/public/cpp/secret.h>


namespace {
    using namespace NYql::NUdf;

    struct TSecretIndexes {
        TType* AdditionalType;
        ui32 SecretIndex, SecretTypeIndex, ValidatedIndex, AdditionalIndex;
        static constexpr int FieldsCount = 4;
    };

    class TSearch: public TBoxedValue {
    public:
        explicit TSearch(const TSecretIndexes& secretIndexes)
                : secretIndexes(secretIndexes)
                , searcher(new NSecretSearch::TSearcher(NSecretSearch::TSearchOptions{
                        .Validate = true,
                        .WithSourceLine = false,
                }))
        {}

        static TStringRef Name()
        {
            static auto name = TStringRef::Of("Search");
            return name;
        }

    private:
        TSecretIndexes secretIndexes;
        THolder<NSecretSearch::TSearcher> searcher;

        TUnboxedValue Run(
                const IValueBuilder* valueBuilder,
                const TUnboxedValuePod* args) const override
        {
            if (!args[0]) {
                return valueBuilder->NewEmptyList();
            }

            try {
                TString inputSecret = TString{args[0].AsStringRef()};
                NSecretSearch::TSourceContent source(inputSecret);
                auto antSecrets = searcher->CheckSource(source);

                if (!antSecrets) {
                    return valueBuilder->NewEmptyList();
                }

                const auto& secrets = antSecrets.GetRef();
                TUnboxedValue* resultList;
                auto result = valueBuilder->NewArray(secrets.size(), resultList);
                for (unsigned i = 0; i < antSecrets->size(); ++i) {
                    const auto& secret = secrets[i];

                    auto additionalInfo = valueBuilder->NewDict(secretIndexes.AdditionalType, 0);
                    for (const auto& [k, v] : secret.Additional) {
                        additionalInfo->Add(valueBuilder->NewString(k), valueBuilder->NewString(v));
                    }

                    TUnboxedValue* structItems;
                    resultList[i] = valueBuilder->NewArray(TSecretIndexes::FieldsCount, structItems);

                    structItems[secretIndexes.SecretIndex] = valueBuilder->NewString(secret.Secret);
                    structItems[secretIndexes.SecretTypeIndex] = valueBuilder->NewString(secret.Type);
                    structItems[secretIndexes.ValidatedIndex] = TUnboxedValuePod(secret.Validated);
                    structItems[secretIndexes.AdditionalIndex] = additionalInfo->Build();
                }

                return result;
            } catch (const yexception &e) {
                UdfTerminate(e.what());
            }
        }
    };

    class TAntSecretModule: public IUdfModule {
    public:
        TStringRef Name() const {
            return TStringRef::Of("AntSecret");
        }

        void CleanupOnTerminate() const final {}

        void GetAllFunctions(IFunctionsSink& sink) const final {
            sink.Add(TSearch::Name());
        }

        void BuildFunctionTypeInfo(
                const TStringRef& name,
                TType* userType,
                const TStringRef& typeConfig,
                ui32 flags,
                IFunctionTypeInfoBuilder& builder) const final
        {
            Y_UNUSED(userType);
            Y_UNUSED(typeConfig);

            if (name != TSearch::Name()) {
                return;
            }

            bool typesOnly = (flags & TFlags::TypesOnly);

            try {
                TSecretIndexes secretIndexes{
                    .AdditionalType = builder.Dict()->Key<char*>().Value<char*>().Build()
                };

                builder
                    .Returns(builder.List()->Item(
                            builder.Struct(TSecretIndexes::FieldsCount)
                                    ->AddField<char*>("secret", &secretIndexes.SecretIndex)
                                    .AddField<char*>("secret_type", &secretIndexes.SecretTypeIndex)
                                    .AddField<bool>("validated", &secretIndexes.ValidatedIndex)
                                    .AddField("additional_info", secretIndexes.AdditionalType, &secretIndexes.AdditionalIndex)
                                    .Build()
                            )
                    )
                    .Args()->Add<TOptional<char*>>().Done();

                if (!typesOnly) {
                    builder.Implementation(new TSearch(secretIndexes));
                }
            }  catch (const yexception& e) {
                builder.SetError(CurrentExceptionMessage());
            }
        }
    };

}

REGISTER_MODULES(TAntSecretModule);
