#include <ydb/library/yql/public/udf/udf_value.h>
#include <ydb/library/yql/public/udf/udf_helpers.h>
#include <ydb/library/yql/public/udf/udf_type_builder.h>

#include <security/ant-secret/internal/validation/validator.h>
#include <security/ant-secret/snooper/internal/secret/secret_types.h>

#include <library/cpp/string_utils/quote/quote.h>

namespace {
    using namespace NYql::NUdf;

    inline TString decodeSecret(const TStringBuf data) {
        // decode urlencoded
        if (data.Contains("%")) {
            return UrlUnescapeRet(data);
        }
        return TString{data};
    }

    class TValidate: public TBoxedValue {
    public:
        static TStringRef Name()
        {
            static auto name = TStringRef::Of("Validate");
            return name;
        }

        static bool DeclareSignature(
                const TStringRef& name,
                TType* /*userType*/,
                IFunctionTypeInfoBuilder& builder,
                bool typesOnly)
        {
            if (Name() == name) {
                builder
                        .Args(2)
                        ->Add<TOptional<ui32>>()
                        .Add<TOptional<char*>>()
                        .Done()
                        .Returns<bool>();

                if (!typesOnly) {
                    builder.Implementation(new TValidate);
                }
                return true;
            } else {
                return false;
            }
        }
    private:
        THolder<NValidation::TValidator> validator;

        TValidate() : validator(new NValidation::TValidator) {}

        TUnboxedValue Run(
                const IValueBuilder* /*valueBuilder*/,
                const TUnboxedValuePod* args) const override
        {
            if (!args[0] || !args[1]) {
                return TUnboxedValuePod(false);
            }

            //TODO(buglloc): do smth better
            bool validated = false;
            try {
                auto const &secretType = NSecret::ESecretType(args[0].Get<ui32>());
                auto const &secret = args[1].AsStringRef();

                switch (secretType) {
                    case NSecret::ESecretType::YOAuth: {
                        auto res = validator->Call("yandex-oauth", decodeSecret(secret));
                        validated = res.Defined() && res->Valid;
                        break;
                    }
                    case NSecret::ESecretType::YSession: {
                        auto res = validator->Call("yandex-session", decodeSecret(secret));
                        validated = res.Defined() && res->Valid;
                        break;
                    }
                    case NSecret::ESecretType::YCApiKey: {
                        auto res = validator->Call("yc-apikey", decodeSecret(secret));
                        validated = res.Defined() && res->Valid;
                        break;
                    }
                    case NSecret::ESecretType::YCToken: {
                        auto res = validator->Call("yc-token", decodeSecret(secret));
                        validated = res.Defined() && res->Valid;
                        break;
                    }
                    case NSecret::ESecretType::YCCookie: {
                        auto res = validator->Call("yc-cookie", decodeSecret(secret));
                        validated = res.Defined() && res->Valid;
                        break;
                    }
                    case NSecret::ESecretType::YCStaticCred:
                        // YC static creds are always invalid
                        validated = false;
                        break;
                    case NSecret::ESecretType::TVMTicket:
                        // TVM tickets are always valid
                    case NSecret::ESecretType::S3Presign:
                        // S3Presign are always valid
                    case NSecret::ESecretType::MdsSign:
                        // MdsSign are always valid
                        validated = true;
                        break;
                    case NSecret::ESecretType::Unknown:
                    case NSecret::ESecretType::All:
                    case NSecret::ESecretType::AllWMds:
                        break;
                }
            } catch (const yexception &e) {
                UdfTerminate(e.what());
            }

            return TUnboxedValuePod(validated);
        }
    };

    SIMPLE_MODULE(TSnooperModule, TValidate);
}

REGISTER_MODULES(TSnooperModule);
