import os
from flask import Flask, Blueprint
from app.database import db
from app.api.for_collectors.endpoints.objects_import import ns as collectors_namespace
from app.api.for_services.endpoints.objects_search import ns as services_namespace

CONFIG_NAME_MAPPER = {
    'development': 'config.Development.cfg',
    'production': 'config.Production.cfg'
}


def create_app(flask_config_name=None):
    """
    Create a global flask instance
    :param flask_config_name: 'development' or 'production' enviroment
    :return: global flask instance
    """

    # load config
    env_flask_config_name = os.getenv('FLASK_CONFIG', 'production')
    if not env_flask_config_name and flask_config_name is None:
        flask_config_name = 'development'
    elif flask_config_name is None:
        flask_config_name = env_flask_config_name
    try:
        if CONFIG_NAME_MAPPER[flask_config_name] is None:
            return None
    except:
        return None

    # create app
    app = Flask(__name__)
    app.url_map.strict_slashes = False
    app.config.from_pyfile(CONFIG_NAME_MAPPER[flask_config_name])

    app.config['SQLALCHEMY_DATABASE_URI'] = os.environ.get('SQLALCHEMY_DATABASE_URI', '')
    app.config['SECRET_KEY'] = os.environ.get('SECRET_KEY', '')

    db.init_app(app)

    # adding api blueprint
    blueprint = Blueprint('api', __name__, url_prefix='/api')
    from app.api import api
    api.init_app(blueprint)
    api.add_namespace(collectors_namespace)
    api.add_namespace(services_namespace)
    app.register_blueprint(blueprint)

    # adding ui blueprint
    from app.ui import ui
    app.register_blueprint(ui)

    # adding idm blueprint
    from app.idm import idm
    app.register_blueprint(idm)

    from app.api.v1 import apiv1
    app.register_blueprint(apiv1)
    return app
