from app.database import db
from app.database.project import Project
from app.database.object import Object
from requests import post
import os
import requests
import urllib.parse
from datetime import datetime
import json
from sqlalchemy.orm.attributes import flag_modified


SECNOTIFY_URL = "https://secnotify.sec.yandex-team.ru/"
SECNOTIFY_TEMPLATE = "discovery_new_object"

def create_st_comment(object, issue):
        url = urllib.parse.urljoin(SECNOTIFY_URL, "/tracker/comment")
        token = os.getenv("discovery_notification_token", "")
        headers = {
            "Authorization": token
        }
        data = {
            "issue_id": issue,
            "template_id": SECNOTIFY_TEMPLATE,
            "template_parameters": {
                "value": object.value,
                "fields": object.fields
            },
        }
        response = requests.post(url, headers=headers, json=data, timeout=5)
        if response.status_code != 200:
            error = response.json()
            print(f'Error sending ST notification ({response.status_code}): {error.get("message") if isinstance(error, dict) else error}')


def send_to_HEC(events):
    """
    Export to SPLUNK function
    :param events: JSON
    :return:
    """
    access_token = os.getenv('SPLUNK_TOKEN', '')
    url = "https://hatch.yandex.net/services/collector"

    auth_header = "Splunk {}".format(access_token)
    headers = {'Authorization': auth_header}

    data = [dict(event=event) for event in events]
    res = post(url, json=json.dumps(data, default=str), headers=headers, verify=False)
    if res.status_code != 200:
        print("Error sending request: status code {}".format(res.status_code))
        return
    json_data = res.json()
    if json_data.get("text") != "Success":
        print("Error sending request:")
        print(json_data)


def export_object(project_id, data_objects):
    project = Project.query.filter(Project.id == project_id).one()
    for object in data_objects:
        object["project_id"] = project_id
        object["project_full_name"] = project.organization.name + " / " + project.name
        object_type = object['type']
        if object_type == 'repo':
            object['queues'] = list()
            for ancestor_id in object['ancestors']:
                ancestor = Object.query.filter_by(uuid=ancestor_id).first()
                if ancestor.type != 'queue':
                    continue
                object['queues'].append(ancestor.value)

    #
    #
    # send_to_HEC(data_objects)


def add_object_to_project(project_id, data_objects):
    project = Project.query.filter(Project.id == project_id).one()
    objects_dict = {object['uuid']: object for object in data_objects}
    # updating existing objects in database
    db_objects = Object.query.filter(Object.uuid.in_(objects_dict.keys())).all()
    for object in db_objects:
        if 'value' in objects_dict[object.uuid]: object.value = objects_dict[object.uuid]['value']
        if 'ancestors' in objects_dict[object.uuid]:
            for ancestor in objects_dict[object.uuid]['ancestors']:
                if ancestor not in object.ancestors:
                    object.ancestors.append(ancestor)
            flag_modified(object, 'ancestors')
        if 'created_at' in objects_dict[object.uuid]: object.created_at = objects_dict[object.uuid]['created_at']
        if 'updated_at' in objects_dict[object.uuid]: object.updated_at = objects_dict[object.uuid]['updated_at']
        for field_name in objects_dict[object.uuid]['fields'].keys():
            object.fields[field_name] = objects_dict[object.uuid]['fields'][field_name]
        flag_modified(object, 'fields')
        db.session.add(object)
        del objects_dict[object.uuid]
    # adding new objects to database
    # similar date for bulk objects
    added_at = datetime.utcnow()
    for object_uuid in objects_dict.keys():
        data_object = objects_dict[object_uuid]
        if 'value' not in data_object: data_object['value'] = ""
        if 'ancestors' not in data_object: data_object['ancestors'] = []
        if 'created_at' not in data_object: data_object['created_at'] = datetime.utcnow()
        if 'updated_at' not in data_object: data_object['updated_at'] = datetime.utcnow()
        object = Object(uuid=object_uuid, type=data_object['type'], project_id=project.id,
                        fields=data_object['fields'], value=data_object['value'], ancestors=data_object['ancestors'],
                        created_at=data_object['created_at'], updated_at=data_object['updated_at'], added_at=added_at)
        if project.enable_notifications:
            create_st_comment(object, project.st_issue)
        db.session.add(object)
    db.session.commit()
    export_object(project_id, data_objects)
    return True
