from cachetools import cached, TTLCache
from requests import get, post
import traceback, sys
from os import getenv
from yaseclib.abc import Abc
from yaseclib.staff import Staff


oauth_token = getenv('OAUTH_TOKEN', '')
awacs_token = getenv('AWACS_TOKEN', '')
robot_creds = getenv('ROBOT_BASIC_AUTH', '')
ml_cache = TTLCache(maxsize=10, ttl=12*3600)
owners_cache = TTLCache(maxsize=10, ttl=3600)
dns_cache = TTLCache(maxsize=10, ttl=3600)
awacs_cache = TTLCache(maxsize=10, ttl=3600)


@cached(awacs_cache)
def get_awacs():
    awacs_results = dict()
    abc = Abc('https://abc-back.yandex-team.ru/api/v3', oauth_token)

    url = 'https://awacs.yandex-team.ru/api/ListNamespaces/'
    headers = {
        'Authorization': 'OAuth {}'.format(awacs_token)
    }
    payload = {
        'fieldMask': "meta"
    }

    res = post(url, json=payload, headers=headers, verify=False)
    if res.status_code != 200:
        return awacs_results

    json_data = res.json()
    namespaces = json_data['namespaces']

    for ns in namespaces:
        meta = ns.get('meta')
        if not meta:
            continue

        host = meta.get('id')

        resps = list()
        auth = meta.get('auth')
        if auth:
            staff = auth.get('staff')
            if staff:
                owners = staff.get('owners')
                if owners:
                    # admins = owners.get('groupIds')
                    resps = owners.get('logins')

        if not resps:
            abcServiceId = meta.get('abcServiceId')
            if abcServiceId:
                resps = abc.get_service_responsibles(abcServiceId)

        awacs_results[host] = resps

    return awacs_results


def get_awacs_admins(fqdn):
    awacs_dict = get_awacs()
    return awacs_dict.get(fqdn, list())

@cached(ml_cache)
def get_ml():
    # FIXME: also look at ml.yandex-team.ru
    url = 'https://abc-back.yandex-team.ru/api/v3/services/contacts/'
    payload = {
        'page_size': 1000,
        'type__code': 'email_ml',
    }
    headers = {
        'Authorization': 'OAuth {}'.format(oauth_token),
    }
    ml_dict = dict()
    page = 1
    total_pages = 1
    while page <= total_pages:
        payload['page'] = page
        res = get(url, params=payload, headers=headers)
        if res.status_code != 200:
            return ml_dict
        json_data = res.json()
        total_pages = json_data['total_pages']
        for item in json_data['results']:
            service = item['service']['slug']
            ml = item['content']
            ml_dict[ml] = service
        page += 1
    return ml_dict


@cached(owners_cache)
def _get_racktables_owners_json():
    url = 'https://noc-export.yandex.net/rt/owners.json'
    res = get(url, verify='/etc/ssl/certs/ca-certificates.crt')
    if res.status_code != 200:
        return dict()
    return res.json()


def _get_racktables_vs_dict():
    json_data = _get_racktables_owners_json()
    if not json_data:
        return dict()
    json_data = json_data.get('vs', dict())
    owners = dict()
    for fqdn, owners_dict in json_data.items():
        owners_dict = owners_dict.get('fw')
        if owners_dict:
            owners[fqdn] = owners_dict
    # remove "svc_qloud_administration" if there are more owners
    # https://st.yandex-team.ru/MOLLY-332#5e417879fa26463935bb1298
    update_owners = dict()
    for fqdn, fqdn_owners in owners.items():
        if len(fqdn_owners) > 1:
            update_owners[fqdn] = list(filter(lambda x:x.get("subj", "")!="svc_qloud_administration", fqdn_owners))
    owners.update(update_owners)
    #
    return owners


def _get_racktables_fws_dict():
    json_data = _get_racktables_owners_json()
    if not json_data:
        return dict()
    json_data = json_data.get('fws', dict())
    owners = dict()
    for fqdn, owners_dict in json_data.items():
        owners_dict = owners_dict.get('fw')
        if owners_dict:
            owners[fqdn] = owners_dict
    return owners


def _parse_racktables_owners_list(owners_list):
    owners = set()
    abc = Abc('https://abc-back.yandex-team.ru/api/v3', oauth_token)
    staff = Staff('https://staff-api.yandex-team.ru/v3', oauth_token)

    for item in owners_list:
        subj_type = item['subj_type']
        subj = item['subj']
        if subj_type == 'user':
            if subj.startswith('robot-'):
                subj = staff.get_robot_responsible(subj)
                if subj is None:
                    continue
            owners.add(subj)
        elif subj_type == 'service':
            service = subj[4:]
            try:
                service_info = abc.get_service_by_slug(service)
                service_id = service_info['id']
                responsibles = abc.get_service_responsibles(service_id)
                owners.update(responsibles)
            except:
                traceback.print_exc(file=sys.stdout)
                continue
        elif subj_type == 'servicerole':
            service = subj.split('_', 2)[1]
            role = subj.split('_', 2)[2]
            try:
                service_info = abc.get_service_by_slug(service)
                service_id = service_info['id']
                responsibles = abc.get_people_by_id(service_id, False, role)
                if not responsibles:
                    responsibles = abc.get_service_responsibles(service_id)
                owners.update(responsibles)
            except:
                traceback.print_exc(file=sys.stdout)
                continue
        elif subj_type == 'department':
            department = subj
            group_info = staff.get_group_info(department, fields=['department.heads'])
            if 'department' not in group_info or not group_info['department'].get('heads'):
                continue
            department_head = group_info['department']['heads'][0]['person']['login']
            owners.add(department_head)

    return list(owners)


def get_racktables_fqdn_owners(fqdn):
    racktables_vs_owners = _get_racktables_vs_dict()
    owners_list = racktables_vs_owners.get(fqdn, list())
    owners = _parse_racktables_owners_list(owners_list)

    return owners


def get_racktables_macros_owners(macros):
    racktables_fws_owners = _get_racktables_fws_dict()
    owners_list = racktables_fws_owners.get(macros, list())
    owners = _parse_racktables_owners_list(owners_list)

    return owners


def resolve_racktables_dnscache_owners(owners_list):
    abc = Abc('https://abc-back.yandex-team.ru/api/v3', oauth_token)
    staff = Staff('https://staff-api.yandex-team.ru/v3', oauth_token)
    owners = set()
    ml_dict = get_ml()

    for item in owners_list:
        if item in ml_dict:
            service = ml_dict[item]
            service_info = abc.get_service_by_slug(service)
            service_id = service_info['id']
            responsibles = abc.get_service_responsibles(service_id)
            owners.update(responsibles)
        else:
            owner = item
            if owner.startswith('robot-'):
                owner = staff.get_robot_responsible(owner)
                if owner is None:
                    continue
            owners.add(owner)

    return owners


@cached(dns_cache)
def get_racktables_dnscache():
    url = 'https://cvs.yandex-team.ru/cgi-bin/cvsweb.cgi/~checkout~/noc/routers/fw/router.dnscache.full'
    headers = {
        'Authorization': 'Basic {}'.format(robot_creds)
    }
    owners = dict(ip=dict(), fqdn=dict())
    res = get(url, headers=headers, verify='/etc/ssl/certs/ca-certificates.crt')
    if res.status_code != 200:
        return owners
    for line in res.text.split('\n'):
        line = line.strip()
        data = line.split(' ')
        if len(data) != 4:
            continue
        fqdn = data[0]
        ips = data[1].split(',')
        owners_list = data[3].split(',')
        owners['fqdn'][fqdn] = owners_list
        for ip in ips:
            owners['ip'][ip] = owners_list

    return owners


def get_cauth_admins(fqdn):
    url = 'https://cauth.yandex.net:4443/passwd/serveradmins/?q={}'.format(fqdn)
    res = get(url, verify='/etc/ssl/certs/ca-certificates.crt')
    if res.status_code != 200:
        return list()

    staff = Staff('https://staff-api.yandex-team.ru/v3', oauth_token)
    owners = set()
    for line in res.text.split('\n'):
        if line.startswith('########') or ':' not in line:
            continue
        admin = line.split(':', 1)[0]
        if admin.startswith('robot-'):
            admin = staff.get_robot_responsible(admin)
            if admin is None:
                continue
        owners.add(admin)
    return list(owners)


def _get_qloud_admins(fqdn, source):
    url = 'https://{}.yandex-team.ru/api/find-object?pattern={}'.format(source, fqdn)
    headers = {
        'Authorization': 'OAuth {}'.format(oauth_token)
    }
    res = get(url, headers=headers, verify='/etc/ssl/certs/ca-certificates.crt')
    if res.status_code != 200:
        return list()
    json_data = res.json()

    env = None
    domains = json_data.get('DOMAIN', list())
    for domain in domains:
        matched = domain.get('matchedObject')
        if fqdn == matched:
            env = domain.get('linkObject', '')
            break
    if not env:
        allocations = json_data.get('ALLOCATION', list())
        for allocation in allocations:
            matched = allocation.get('matchedObject')
            if fqdn == matched:
                env = allocation.get('linkObject', '')
                env = env.rsplit('.', 1)[0]
                break
    if not env:
        return list()

    url = 'https://{}.yandex-team.ru/api/v1/grants/{}'.format(source, env)
    res = get(url, headers=headers, verify='/etc/ssl/certs/ca-certificates.crt')
    if res.status_code != 200:
        return list()

    staff = Staff('https://staff-api.yandex-team.ru/v3', oauth_token)
    json_data = res.json()
    acls = json_data.get('acls', list())
    admins_acls = list(filter(lambda acl: acl['permission'] == 'ADMINISTRATOR', acls))

    resp_acl = list(filter(lambda acl: acl['grantedObjectLevel'] == 'Environment', admins_acls))
    if not resp_acl:
        resp_acl = list(filter(lambda acl: acl['grantedObjectLevel'] == 'Application', admins_acls))
    if not resp_acl:
        resp_acl = list(filter(lambda acl: acl['grantedObjectLevel'] == 'Project', admins_acls))

    # Collect persons and robots separately.
    owners = set()
    owners_of_robots = set()
    for acl in resp_acl:
        admin = acl['user']
        if admin.startswith('robot-'):
            owners_of_robots.add(admin)
        else:
            owners.add(admin)
    # Dont resolve rebot owners if persons have been found
    if not owners:
        for admin in owners_of_robots:
            robot_resp = staff.get_robot_responsible(admin)
            if robot_resp:
                owners.add(robot_resp)

    return list(owners)


def get_qloud_admins(fqdn):
    sources = ['platform', 'platform-int']
    admins = list()
    for source in sources:
        admins = _get_qloud_admins(fqdn, source)
        if admins:
            break
    return admins


def get_fw_macros(ip):
    url = 'https://ro.racktables.yandex-team.ru/export/get-net-by-ip.php?ip={}'.format(ip)
    res = get(url)
    if res.status_code != 200 or res.text.startswith('Usage:') or res.text == 'Network not found':
        return None
    macros = res.text.strip().split('\t')[-1][1:-1]

    return macros


def get_owners(data=None, data_type='fqdn'):
    if data_type == 'fqdn':
        fqdn = data

        owners_list = get_qloud_admins(fqdn)
        if owners_list:
            return dict(owners=owners_list, source='qloud')

        owners_list = get_cauth_admins(fqdn)
        if owners_list:
            return dict(owners=owners_list, source='cauth')

        # racktables dnscache
        racktables_dnscache = get_racktables_dnscache()
        owners_list = racktables_dnscache['fqdn'].get(fqdn, list())
        owners = resolve_racktables_dnscache_owners(owners_list)

        if owners:
            return dict(owners=list(owners), source='dnscache')

        owners_list = get_awacs_admins(fqdn)
        if owners_list:
            return dict(owners=owners_list, source='awacs')

        owners_list = get_racktables_fqdn_owners(fqdn)
        if owners_list:
            return dict(owners=owners_list, source='racktables')

    elif data_type == 'ip':
        ip = data

        # racktables dnscache
        racktables_dnscache = get_racktables_dnscache()
        owners_list = racktables_dnscache['ip'].get(ip, list())
        owners = resolve_racktables_dnscache_owners(owners_list)

        if owners:
            return dict(owners=list(owners), source='dnscache')

        # get fw macros and search for owners
        macros = get_fw_macros(ip)
        if not macros:
            return None
        owners_list = get_racktables_macros_owners(macros)
        if owners_list:
            return dict(owners=owners_list, macros=macros, source='racktables')

    return None
