from flask import request, jsonify
from flask_restplus import Resource
from app.api import api
from app.api.for_services.business import search_child_objects, search_ancestor_object, search_objects
from app.api.for_services.debby_helper import get_owners
from werkzeug.exceptions import NotFound
from app import yauth
import sys
import traceback

ns = api.namespace('for_services', description='Operations from services')


@ns.route('/search_host_owner')
class ChildObjectSearch(Resource):
    def post(self):
        """
        Search host owners
        {
            "fqdn":"ios-build-geo01.techadmin.yandex.net"
        }
        """
        fqdn = request.json.get('fqdn')
        ip = request.json.get('ip')
        if fqdn:
            data = fqdn
            data_type = 'fqdn'
        elif ip:
            data = ip
            data_type = 'ip'
        else:
            return {"error": {"message": "Wrong request, use fqdn or ip"}}, 400

        try:
            result = get_owners(data, data_type)
            if result:
                return {"results": result}, 200
            else:
                return {"error": {"message": "Not found"}}, 404
        except:
            traceback.print_exc(file=sys.stdout)
            return {"error": {"message": "Something goes wrong"}}, 500


@ns.route('/search_childs')
class ChildObjectSearch(Resource):
    def post(self):
        """
        Search child-objects
        {
            "ancestor_type":"service",
            "ancestor_field_name":"slug",
            "ancestor_field_value":"taxi",
            "child_type":"queue",
            "recursive":true
        }
        """
        data = request.json
        try:
            child_objects = search_child_objects(data["ancestor_type"], data["ancestor_field_name"],
                             data["ancestor_field_value"], data["child_type"], data.get("recursive", True))
            print(map(lambda x:x, child_objects))
            return {"message": "OK", "objects": list(map(lambda obj: obj.to_json(), child_objects))}, 200
        except NotFound as exp:
            return {"message": "Ancestor Object not found"}, 404
        # "objects": map(lambda obj: obj.fields, child_objects)


@ns.route('/search_ancestors')
class AncestorObjectSearch(Resource):
    def post(self):
        """
        Search ancestor object
        {
            "child_type":"queue",
            "child_value": "TAXISECURITY",
            "ancestor_type":"service"
        }
        or
        {
            "child_type":"queue",
            "child_field_name":"slug",
            "child_field_value":"taxi",
            "ancestor_type":"service"
        }
        """
        data = request.json
        try:
            if "child_value" in data:
                ancestors = search_ancestor_object(child_type=data["child_type"], ancestor_type=data["ancestor_type"],
                                                   child_value=data["child_value"])
            else:
                ancestors = search_ancestor_object(child_type=data["child_type"], ancestor_type=data["ancestor_type"],
                                                   child_field_name=data["child_field_name"],
                                                   child_field_value=data["child_field_value"])
            if ancestors:
                return {"message": "OK", "ancestors": list(map(lambda obj: obj.to_json(), ancestors))}, 200
            else:
                return {"message": "NOT_FOUND", "ancestor": []}, 200
        except NotFound as exp:
            return {"message": "Child object not found"}, 404



@ns.route('/object_search/<project_id_or_slug>')
class ObjectSearch(Resource):
    @yauth.apikey_required_for_project()
    def get(self, project_id_or_slug):
        """
        /api/for_services/object_search/1/?start_date=2019-01-20&finish_date=2019-01-21&event_type=release
        """
        start_date = request.args.get('start_date', default=None)
        finish_date = request.args.get('finish_date', default=None)
        event_type = request.args.get('event_type', default=None)
        checked = request.args.get('checked', default=False)
        results = search_objects(project_id_or_slug, start_date, finish_date, event_type, checked)
        return {"code": 200, "result": results}, 200


@ns.route('/object_duty_search/<int:project_id>')
class ObjectDutySearch(Resource):
    # 2010090 - C3P0 TVM_ID
    @yauth.tvm_ticket_required(2010090)
    def get(self, project_id):
        """
        /object_duty_search/?start_date=2019-01-20&finish_date=2019-01-21&event_type=release
        """
        start_date = request.args.get('start_date', default=None)
        finish_date = request.args.get('finish_date', default=None)
        event_type = request.args.get('event_type', default=None)
        checked = request.args.get('checked', default=False)
        results = search_objects(project_id, start_date, finish_date, event_type, checked)
        return {"code": 200, "result": results}, 200
