from flask import render_template, redirect, send_from_directory, request
import flask
import os
from app.ui import ui
from app.database.organization import Organization
from app.database.project import Project
from app.database.scheduler import Scheduler
from app.database.object import Object
from app.database.user import User
from app.ui.forms import OrganizationForm, ProjectForm, SanboxTaskForm
from app.database import db
from app.sandbox_api.sandbox_api import SandboxApi
from app.sandbox_api.sandbox_discovery import SandboxDiscovery
from sqlalchemy import Date, cast, or_, and_
from app import yauth


@ui.route('/favicon.ico')
def favicon():
    return send_from_directory(os.path.join(ui.root_path, 'static'),
                               'favicon.ico', mimetype='image/vnd.microsoft.icon')


@ui.route('/js/<path:path>')
def send_js(path):
    return send_from_directory(os.path.join(ui.root_path, 'static', 'js'), path)


@ui.route('/css/<path:path>')
def send_css(path):
    return send_from_directory(os.path.join(ui.root_path, 'static', 'css'), path)


@ui.route('/')
@yauth.sessionid_required
@yauth.idm_role_required('admin', 'security', 'read')
def show_all():
    # get all allowed organization to view
    user_roles = User.query.filter(User.login == flask.g.yalogin).all()
    user_view_filters = []
    show_all = False
    for user_role in user_roles:
        if user_role.organization_slug == '*':
            user_view_filters = []
            show_all = True
            break
        else:
            user_view_filters.append(Organization.slug.ilike(user_role.organization_slug))
    if show_all:
        organizations = Organization.query.order_by(Organization.name).all()
    else:
        organizations = Organization.query.filter(or_(*user_view_filters)).order_by(Organization.name).all()
    is_global_admin = yauth.check_allowed_roles_for_current_user(['admin'], '*', '*')
    return render_template('show_all.html', organizations=organizations, is_admin=is_global_admin)


@ui.route('/registry/<organization_id>')
@yauth.sessionid_required
@yauth.idm_role_required('admin', 'security', 'read')
def view_organization(organization_id):
    organization = Organization.query.filter_by(id=organization_id).first()
    # get all allowed projects to view
    user_roles = User.query.filter(and_(User.login == flask.g.yalogin,
                                        or_(User.organization_slug=='*', User.organization_slug==organization.slug))).all()
    user_view_filters = []
    show_all = False
    for user_role in user_roles:
        if user_role.organization_slug == '*' or user_role.project_slug == '*':
            user_view_filters = []
            show_all = True
            break
        else:
            user_view_filters.append(Project.slug.ilike(user_role.project_slug))
    if show_all:
        projects = Project.query.filter(Project.organization_id == organization.id).order_by(Project.name).all()
    else:
        projects = Project.query.filter(Project.organization_id == organization.id).filter(or_(*user_view_filters)).order_by(Project.name).all()
    is_organization_admin = yauth.check_allowed_roles_for_current_user(['admin'], organization.slug, '*')
    return render_template('show_organization.html', organization=organization, projects=projects,
                           is_admin=is_organization_admin)


@ui.route('/organization/<int:organization_id>/edit', methods=('GET', 'POST'))
@yauth.sessionid_required
@yauth.idm_role_required('admin')
def edit_organization(organization_id):
    organization = Organization.query.get_or_404(organization_id)
    form = OrganizationForm(obj=organization)
    if form.validate_on_submit():
        organization.name = form.name.data
        organization.slug = form.slug.data
        if form.apikey.data:
            organization.apikey = form.apikey.data
        db.session.add(organization)
        db.session.commit()
        return redirect('/registry/' + str(organization.id))
    else:
        return render_template('edit_organization.html', form=form, organization=organization)


@ui.route('/add_organization', methods=('GET', 'POST'))
@yauth.sessionid_required
@yauth.idm_role_required('admin')
def add_organization():
    form = OrganizationForm()
    if form.validate_on_submit():
        organization = Organization(form.name.data, form.slug.data, form.apikey.data)
        db.session.add(organization)
        db.session.commit()
        return redirect('/registry/'+str(organization.id))
    else:
        return render_template('add_organization.html', form=form)


@ui.route('/registry/projects/<project_id>')
@yauth.sessionid_required
@yauth.idm_role_required('admin', 'security', 'read')
def view_project(project_id):
    project = Project.query.filter_by(id=project_id).first()
    is_project_admin = yauth.check_allowed_roles_for_current_user(['admin'], project.organization.slug, project.slug)
    return render_template('show_project.html', project=project,
                           json_persons=list(map(lambda x: {"text": str(x), "href": x.href_link()},
                                                 project.persons.order_by(Object.created_at.desc()).limit(100))),
                           json_services=list(map(lambda x: {"text": str(x), "href": x.href_link()},
                                                  project.services.order_by(Object.created_at.desc()).limit(100))),
                           json_queues=list(map(lambda x: {"text": str(x), "href": x.href_link()},
                                                project.queues.order_by(Object.created_at.desc()).limit(100))),
                           json_repos=list(map(lambda x: {"text": str(x)},
                                               project.repositories.order_by(Object.created_at.desc()).limit(100))),
                           json_commits=list(map(lambda x: {"text": str(x), "href": x.href_link()},
                                                 project.commits.order_by(Object.created_at.desc()).limit(100))),
                           json_releases=list(map(lambda x: {"text": str(x), "href": x.href_link()},
                                                  project.releases.order_by(Object.created_at.desc()).limit(100))),
                           json_handles=list(map(lambda x: {"text": str(x), "href": x.href_link()},
                                                 project.handles.order_by(Object.created_at.desc()).limit(100))),
                           json_resources=list(map(lambda x: {"text": str(x)},
                                                   project.resources.order_by(Object.created_at.desc()).limit(100))),
                           is_admin = is_project_admin
                           )


@ui.route('/feed/projects/<project_id>')
@yauth.sessionid_required
@yauth.idm_role_required('admin', 'security', 'read')
def view_project_feed(project_id):
    project = Project.query.filter_by(id=project_id).first_or_404()
    # make filters for feed
    filters = []
    filter_types = []
    check_all = len([c.__name__ for c in Object.__subclasses__() if request.args.get("filter_"+c.__name__)]) == 0
    for c in Object.__subclasses__():
        filter_name = "filter_"+c.__name__
        checked = (request.args.get(filter_name) == 'on') or check_all
        filters.append({'name':filter_name, "color":c.feed_color(), "caption":c.__name__, "checked": checked})
        if checked: filter_types.append(c.__mapper_args__['polymorphic_identity'])
    filter_date = request.args.get('date_filter') or ""
    object_on_page = 100
    page = int(request.args.get('page')) if request.args.get('page') else 0
    objects = Object.query.filter_by(project_id=project.id).filter(Object.type.in_(filter_types))
    if filter_date: objects = objects.filter(cast(Object.created_at, Date) == filter_date)
    objects = objects.order_by(Object.added_at.desc(), Object.created_at.desc()).offset(object_on_page*page)\
        .limit(100).all()
    is_project_security = yauth.check_allowed_roles_for_current_user(['admin', 'security'],
                                                                     project.organization.slug, project.slug)
    return render_template('show_project_feed.html', project = project, objects = objects, current_page = page,
                           filters = filters, filter_date = filter_date, is_security = is_project_security)


@ui.route('/organization/<organization_id>/add_project', methods=('GET', 'POST'))
@yauth.sessionid_required
@yauth.idm_role_required('admin')
def add_project(organization_id):
    form = ProjectForm()
    organization = Organization.query.filter_by(id=organization_id).first()
    if form.validate_on_submit():
        project = Project(form.name.data, form.slug.data,
                          organization.id, form.enable_notifications.data,
                          form.st_issue.data)
        db.session.add(project)
        db.session.commit()
        return redirect('/registry/'+str(organization.id))
    else:
        form.name.data = form.name.data or organization.name
        form.slug.data = form.slug.data or organization.slug
        return render_template('add_project.html', form=form, organization=organization)


@ui.route('/project/<int:project_id>/edit', methods=('GET', 'POST'))
@yauth.sessionid_required
@yauth.idm_role_required('admin')
def edit_project(project_id):
    form = ProjectForm()
    project = Project.query.get_or_404(project_id)
    if form.validate_on_submit():
        project.name = form.name.data
        project.slug = form.slug.data
        project.enable_notifications = form.enable_notifications.data
        project.st_issue = form.st_issue.data
        db.session.add(project)
        db.session.commit()
        return redirect('/registry/projects/'+str(project.id))
    else:
        form.name.data = project.name
        form.slug.data = project.slug
        form.enable_notifications.data = project.enable_notifications
        form.st_issue.data = project.st_issue
        return render_template('edit_project.html', form=form, project=project)


@ui.route('/project/<int:project_id>/add_scheduler', methods=('GET', 'POST'))
@yauth.sessionid_required
@yauth.idm_role_required('admin')
def add_task_to_project(project_id):
    form = SanboxTaskForm()
    project = Project.query.get_or_404(project_id)
    if form.validate_on_submit():
        sandbox_api = SandboxApi()
        scheduler_id = sandbox_api.create_scheduler(project_id=project.id, description=form.description.data,
                                                    pipeline_str = form.pipeline.data,
                                                    input_params_str=form.input_params.data,
                                                    optional_params_str=form.optional_params.data)
        scheduler = Scheduler(project_id=project.id, scheduler_id=scheduler_id, description=form.description.data)
        db.session.add(scheduler)
        db.session.commit()
        return redirect('/project/%d/edit' % project.id)
    else:
        return render_template('add_sandbox_task.html', form=form,
                               project=project, collectors=SandboxDiscovery().get_collectors_list(),
                               collector_params=SandboxDiscovery().get_collector_params())


@ui.route('/project/<int:project_id>/schedulers')
@yauth.sessionid_required
@yauth.idm_role_required('admin')
def view_project_schedulers(project_id):
    project = Project.query.get_or_404(project_id)
    return render_template('show_project_schedulers.html', project=project)


@ui.route('/project/<int:project_id>/object/<object_uuid>/toogle_status', methods=["PUT"])
@yauth.sessionid_required
@yauth.idm_role_required('admin', 'security')
def mark_object_as_checked(project_id, object_uuid):
    project = Project.query.get_or_404(project_id)
    object = Object.query.filter_by(uuid = object_uuid).filter_by(project_id = project.id).first_or_404()
    object.checked = not (object.checked)
    for child in object.get_childs():
        if child.checked != object.checked:
            child.checked = object.checked
            child.last_checked_by = flask.g.yalogin
            db.session.add(child)
    object.last_checked_by = flask.g.yalogin
    db.session.add(object)
    db.session.commit()
    return 'OK'


@ui.route('/object/<object_uuid>/show')
@yauth.sessionid_required
@yauth.idm_role_required('admin', 'security', 'read')
def view_object(object_uuid):
    object = Object.query.filter_by(uuid=object_uuid).first_or_404()
    is_project_security = yauth.check_allowed_roles_for_current_user(['admin', 'security'],
                                    object.project.organization.slug, object.project.slug)
    return render_template('show_object.html', object=object, is_security=is_project_security)