import functools
import flask
import os
import blackbox
import requests
from flask import current_app, request
from app.database import db
from app.database.project import Project
from app.database.organization import Organization
from app.database.user import User
from app.database.object import Object


def tvm_ticket_required(*allowed_src):
    def wrapper(func):
        @functools.wraps(func)
        def wrapped(*args, **kwargs):
            if current_app.config.get('USE_TVM_AUTH'):
                tvm_url = 'http://localhost:2/tvm/checksrv?dst=' + str(current_app.config.get('TVM_ID'))
                ya_service_ticket = request.headers.get('X-Ya-Service-Ticket', '')
                auth_header = os.environ.get('TVMTOOL_LOCAL_AUTHTOKEN', '')
                r = requests.get(tvm_url, headers={'Authorization': auth_header, 'X-Ya-Service-Ticket': ya_service_ticket})
                if r.status_code == 200:
                    tvm_response = r.json()
                    if tvm_response['src'] not in allowed_src:
                        return flask.abort(403)
                else:
                    return flask.abort(403)
            return func(*args, **kwargs)
        return wrapped

    return wrapper


def apikey_required_for_project():
    def wrapper(func):
        @functools.wraps(func)
        def wrapped(*args, **kwargs):
            project_id_or_slug = kwargs['project_id_or_slug']
            if project_id_or_slug:
                if project_id_or_slug.isdigit():
                    project = db.session.query(Project).filter(Project.id == project_id_or_slug).first_or_404()
                else:
                    project = db.session.query(Project).filter(Project.slug == project_id_or_slug).first_or_404()
                apikey = request.headers.get('X-API-Key')
                if apikey and apikey == project.organization.apikey:
                    return func(*args, **kwargs)
                else:
                    return flask.abort(403)
            else:
                return flask.abort(404)

        return wrapped

    return wrapper


def idm_role_required(*allowed_role):
    def wrapper(func):
        @functools.wraps(func)
        def wrapped(*args, **kwargs):
            object = Object.query.filter_by(uuid=kwargs['object_uuid']).first_or_404() if 'object_uuid' in kwargs else None
            if 'project_id' in kwargs:
                project = Project.query.get_or_404(kwargs['project_id'])
            elif object is not None:
                project = object.project
            else:
                project = None
            if 'organization_id' in kwargs:
                organization = Organization.query.get_or_404(kwargs['organization_id'])
            elif project is not None:
                organization = project.organization
            else:
                organization = None
            project_slug = project.slug if project else None
            organization_slug = organization.slug if organization else None
            allowed = check_allowed_roles_for_current_user(allowed_role, organization_slug, project_slug)
            if allowed:
                return func(*args, **kwargs)
            else:
                return flask.abort(403)

        return wrapped

    return wrapper


def check_allowed_roles_for_current_user(allowed_role=[], organization_slug='*', project_slug='*'):
    allowed = False
    user_roles = User.query.filter(User.login == flask.g.yalogin).all()
    for user_role in user_roles:
        if organization_slug:
            if project_slug:
                if user_role.role in allowed_role \
                        and (user_role.organization_slug == '*' \
                             or (user_role.organization_slug == organization_slug \
                                 and user_role.project_slug == '*') \
                             or (user_role.organization_slug == organization_slug \
                                 and user_role.project_slug == project_slug)):
                    allowed = True
            else:
                if user_role.role in allowed_role \
                        and (user_role.organization_slug == '*' \
                             or user_role.organization_slug == organization_slug):
                    allowed = True
        else:
            if user_role.role in allowed_role:
                allowed = True
    return allowed


def get_tvm_ticket(dst_tvm_id):
    url = 'http://localhost:2/tvm/tickets'
    auth_header = os.environ.get('TVMTOOL_LOCAL_AUTHTOKEN', '')
    headers = {'Authorization': auth_header}
    params = {'src': current_app.config.get('TVM_ID'), 'dsts': dst_tvm_id}
    r = requests.get(url, headers=headers, params=params)
    return r.json()


def sessionid_required(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        if current_app.config.get('USE_BLACKBOX_AUTH'):
            validator = _SessionidValidator()
            blackbox_response = validator.validate_sessionid_from_request()
            if validator.blackbox_response_is_valid(blackbox_response):
                flask.g.yalogin = validator.login_from(blackbox_response)
                flask.g.yauid = validator.uid_from(blackbox_response)
                return func(*args, **kwargs)
            else:
                return flask.redirect(validator.get_passport_auth_url())
        else:
            flask.g.yalogin = 'localhost_user'
            return func(*args, **kwargs)

    return wrapper


class _SessionidValidator(object):
    passport_auth_url = 'https://passport.yandex-team.ru/auth?retpath=http%3a%2f%2fappsecdiscovery-test.yandex-team.ru'

    def __init__(self):
        self.blackbox = blackbox.JsonBlackbox(url='http://blackbox.yandex-team.ru/blackbox')

    def get_passport_auth_url(self):
        return self.passport_auth_url

    @staticmethod
    def uid_from(response):
        return int(response['uid']['value'])

    @staticmethod
    def login_from(response):
        return response['login']

    @staticmethod
    def _invalid_status(response):
        status_value = response['status']['value']
        return status_value != 'VALID'

    @staticmethod
    def _error_in(response):
        return 'exception' in response

    @staticmethod
    def blackbox_response_is_valid(blackbox_response):
        return not (blackbox_response is None
                    or _SessionidValidator._error_in(blackbox_response)
                    or _SessionidValidator._invalid_status(blackbox_response))

    @staticmethod
    def _get_userip():
        return flask.request.environ.get('HTTP_X_REAL_IP', 'unknown')

    @staticmethod
    def _get_sessionid():
        return flask.request.cookies.get('Session_id')

    @staticmethod
    def _get_sslsessionid():
        return flask.request.cookies.get('sessionid2')

    @staticmethod
    def _sessionid_not_set():
        return not (_SessionidValidator._get_sessionid() and _SessionidValidator._get_sslsessionid())

    def validate_sessionid_from_request(self):
        if self._sessionid_not_set():
            return None
        try:
            tvm_ticket = get_tvm_ticket(current_app.config.get('BLACKBOX_TVM_ID'))['blackbox']['ticket']
            response = self.blackbox.sessionid(
                userip=self._get_userip(),
                sessionid=self._get_sessionid(),
                host=flask.request.host,
                sslsessionid=self._get_sslsessionid(),
                headers={'X-Ya-Service-Ticket': tvm_ticket})
        except blackbox.BlackboxResponseError as exc:
            return None
        else:
            return response