package main

import (
	"fmt"
	"log"
	"net/http"
	"os"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver/grpcresolver"
	"a.yandex-team.ru/security/burp-collaborator-fetcher/internal/collaborator"
)

const (
	burpEndpointID = "burp-collaborator-http"
)

func main() {
	e := echo.New()

	// YP service discovery
	r, err := grpcresolver.New()
	if err != nil {
		log.Fatalln(err)
	}

	// TODO(buglloc): move to config
	clusters := []string{resolver.ClusterSAS, resolver.ClusterVLA, resolver.ClusterMAN}

	e.GET("/ping", func(c echo.Context) error {
		return c.String(200, "pong")
	})

	e.GET("/burpresults", func(c echo.Context) error {
		responses := make([]*resolver.ResolveEndpointsResponse, 0, len(clusters))
		for _, cluster := range clusters {
			response, err := r.ResolveEndpoints(c.Request().Context(), cluster, burpEndpointID)
			if err != nil {
				log.Printf("failed to discoverer burp: %v\n", err)
				return err
			}
			responses = append(responses, response)
		}

		var upstreams []string
		for _, resp := range responses {
			for _, endpoints := range resp.EndpointSet.Endpoints {
				upstreams = append(upstreams, fmt.Sprintf("[%s]:%d", endpoints.IPv6.String(), endpoints.Port))
			}
		}

		res := collaborator.FetchAll(upstreams, c.QueryString())
		c.Response().Header().Set("X-Collaborator-Version", "4")

		now := time.Now().UnixNano() / (int64(time.Millisecond) / int64(time.Nanosecond))
		c.Response().Header().Set("X-Collaborator-Time", strconv.FormatInt(now, 10))

		return c.JSON(http.StatusOK, res)
	})

	port := os.Getenv("HTTP_PORT")
	if port == "" {
		port = "3000"
	}

	err = e.Start(fmt.Sprintf(":%s", port))
	if err != nil {
		log.Fatalln(err)
	}
}
