package com.yandex.burp.extensions.plugins.audit;

import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IHttpService;
import burp.IRequestInfo;
import burp.IResponseInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;

public class ActuatorPlugin implements IAuditPlugin {

    private static final String SIGNATURE = "_links";
    private static final int ISSUE_TYPE = 0x080a0069;
    private static final String ISSUE_NAME = "Spring Boot Actuator available";
    private static final String SEVERITY = "High";
    private static final String CONFIDENCE = "Certain";
    private static final String POSTFIX = "actuator/";
    private final IBurpExtenderCallbacks callbacks;
    private final IExtensionHelpers helpers;

    public ActuatorPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfig) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
    }

    @Override
    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
        byte insertionPointType = insertionPoint.getInsertionPointType();

        if (insertionPointType == IScannerInsertionPoint.INS_URL_PATH_FILENAME
            || insertionPointType == IScannerInsertionPoint.INS_URL_PATH_FOLDER)
                return null;

        IHttpService httpService = baseRequestResponse.getHttpService();
        IRequestInfo request = this.helpers.analyzeRequest(httpService, baseRequestResponse.getRequest());

        URL requestUrl = request.getUrl();
        String path = normalizePath(requestUrl.getPath());
        if (requestUrl.getPath().contains("."))
            return null;

        IHttpRequestResponse attackRequest = tryRequest(request, httpService, path, POSTFIX);
        IResponseInfo attackResponse = helpers.analyzeResponse(attackRequest.getResponse());

        if (attackRequest.getResponse() == null)
            return null;

        String body = helpers.bytesToString(attackRequest.getResponse());

        if (attackResponse.getStatusCode() == 200
            && attackResponse.getInferredMimeType().equals("JSON")
            && body.contains(SIGNATURE))
                return generateIssues(request, attackRequest);

        return null;
    }

    private String normalizePath(String path) {
        if (!path.endsWith("/")) {
            return path + "/";
        }
        return path;
    }

    private List<IScanIssue> generateIssues(IRequestInfo request, IHttpRequestResponse attackRequest) {
        List<IScanIssue> issues = new ArrayList<>();

        IRequestInfo attackRequestParsed = this.helpers.analyzeRequest(attackRequest);

        String attackDetails = "Spring Boot Actuator available at " + attackRequestParsed.getUrl().toString();

        issues.add(new CustomScanIssue(
                attackRequest.getHttpService(),
                attackRequestParsed.getUrl(),
                new IHttpRequestResponse[]{
                        this.callbacks.applyMarkers(attackRequest,
                                null,
                                null)
                },
                attackDetails,
                ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE,
                "", "", "")
        );

        return issues;
    }

    private IHttpRequestResponse tryRequest(IRequestInfo request, IHttpService httpService, String path, String postfix) {
        String payloadPath = path + postfix;
        String requestLine = "GET " + payloadPath + " HTTP/1.1";

        List<String> headers = request.getHeaders();
        headers.set(0, requestLine);

        byte[] attackBody = helpers.buildHttpMessage(headers, helpers.stringToBytes(""));

        return this.callbacks.makeHttpRequest(httpService, attackBody);
    }
}
