package com.yandex.burp.extensions.plugins.audit;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import burp.IBurpExtenderCallbacks;
import burp.IExtensionHelpers;
import burp.IHttpRequestResponse;
import burp.IHttpService;
import burp.IRequestInfo;
import burp.IScanIssue;
import burp.IScannerInsertionPoint;
import com.yandex.burp.extensions.plugins.CustomScanIssue;
import com.yandex.burp.extensions.plugins.config.BurpMollyPackConfig;


/**
 * Created by ezaitov on 06/09/2020.
 */
public class FlaskDebugPlugin implements IAuditPlugin {

    private static final int ISSUE_TYPE = 0x080a0066;
    private static final String ISSUE_NAME = "Flask in Debug Mode";
    private static final String SEVERITY = "High";
    private static final String CONFIDENCE = "Certain";

    // javascript source: https://github.com/pallets/werkzeug/blob/master/src/werkzeug/debug/shared/debugger.js
    private static final String DEBUGGER_JS_SIGNATURE = "!EVALEX_TRUSTED";
    // if debugger enable should return javascript file
    private static final String PAYLOAD = "?__debugger__=yes&cmd=resource&f=debugger.js";

    private final IBurpExtenderCallbacks callbacks;
    private final IExtensionHelpers helpers;
    private final Set<String> checkedPaths = new HashSet<>();

    public FlaskDebugPlugin(IBurpExtenderCallbacks callbacks, BurpMollyPackConfig extConfig) {
        this.callbacks = callbacks;
        this.helpers = callbacks.getHelpers();
    }

    public List<IScanIssue> doScan(IHttpRequestResponse baseRequestResponse, IScannerInsertionPoint insertionPoint) {
        if ((insertionPoint.getInsertionPointType() != IScannerInsertionPoint.INS_PARAM_NAME_URL))
            return null;

        List<IScanIssue> issues = new ArrayList<>();
        IHttpService httpService = baseRequestResponse.getHttpService();
        IRequestInfo request = this.helpers.analyzeRequest(httpService, baseRequestResponse.getRequest());

        if (request.getUrl() == null)
            return null;

        if (checkedPaths.contains(request.getUrl().getPath()))
            return null;

        String requestLine = "GET " + request.getUrl().getPath() + PAYLOAD + " HTTP/1.1";

        List<String> headers = request.getHeaders();
        headers.set(0, requestLine);
        byte[] attackBody = helpers.buildHttpMessage(headers, helpers.stringToBytes(""));
        IHttpRequestResponse testRequest = this.callbacks.makeHttpRequest(httpService, attackBody);

        if (testRequest.getResponse() == null)
            return null;

        checkedPaths.add(request.getUrl().getPath());

        String body = helpers.bytesToString(testRequest.getResponse());

        if (body.contains(DEBUGGER_JS_SIGNATURE)) {
            List<int[]> requestMarkers = new ArrayList<>(1);
            requestMarkers.add(insertionPoint.getPayloadOffsets(PAYLOAD.getBytes()));

            String attackDetails = "Flask is in debug mode at " +
                    request.getUrl().toString() +
                    ". Needs manual checking if exploitable";

            issues.add(new CustomScanIssue(
                    testRequest.getHttpService(),
                    request.getUrl(),
                    new IHttpRequestResponse[]{this.callbacks.applyMarkers(testRequest, requestMarkers, null)},
                    attackDetails,
                    ISSUE_TYPE, ISSUE_NAME, SEVERITY, CONFIDENCE,
                    "", "", "")
            );
        }
        return issues.isEmpty() ? null : issues;
    }
}


